# FSCompositePage - Composite Pages as Filesystem Objects
# Copyright (C) 2004 Enfold Systems, LLC
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
"""
$Id: fscomposite.py,v 1.8 2005/04/27 01:09:33 dreamcatcher Exp $
"""
from os import path, stat

import Globals
from AccessControl import ClassSecurityInfo
from OFS.Cache import Cacheable
from Acquisition import aq_base, aq_parent

from Products.CompositePage.composite import CompositeMixin, Composite
from Products.CompositePage.composite import SlotCollection
from Products.CMFCore import CMFCorePermissions
from Products.CMFCore.FSObject import FSObject
from Products.CMFCore.FSMetadata import CMFConfigParser
from Products.CMFCore.DirectoryView import registerFileExtension
from Products.CMFCore.DirectoryView import registerMetaType
from Products.CMFCore.utils import expandpath, getToolByName
from Products.CMFCore.CMFCorePermissions import View, ModifyPortalContent
from Products.CMFCore.CMFCorePermissions import ViewManagementScreens
from Products.PageTemplates.PageTemplateFile import PageTemplateFile
from Products.FSCompositePage.element import BoundCompositeElement

class FSCompositePage(CompositeMixin, FSObject, Cacheable):
    """ A Filesystem Composite Page implementation """

    meta_type = 'Filesystem Composite Page'
    _proxy_roles = ()
    _owner = None  # Unowned

    manage_options=(
        ({'label':'Customize', 'action':'manage_main'},)
        + CompositeMixin.manage_options
        + FSObject.manage_options
        + Cacheable.manage_options
        )


    # Use declarative security
    security = ClassSecurityInfo()
    security.declareObjectProtected(View)
    security.declareProtected(View, 'index_html',)

    security.declareProtected(ViewManagementScreens, 'manage_main')
    manage_main = PageTemplateFile('www/customize_composite', globals())


    __init__ = FSObject.__init__

    def _createZODBClone(self):
        """Create a ZODB (editable) equivalent of this object."""
        obj = Composite()
        obj._setId(self.getId())
        obj.title = self.title
        obj.template_path = self.template_path
        if 'filled_slots' in obj.objectIds():
            obj.manage_delObjects(ids=['filled_slots'])
        obj._setObject('filled_slots', aq_base(self.filled_slots))
        # Now we reparse, so that the reference is cleared.
        del self.filled_slots
        self._parsed = 0
        self._updateFromFS()
        return obj

    def _readFile(self, reparse):
        """Read the data from the filesystem.

        Read the file (indicated by exandpath(self._filepath), and parse the
        data if necessary.
        """

        if not getattr(aq_base(self), 'filled_slots', None):
            # Create Slots Container
            f = SlotCollection()
            f._setId("filled_slots")
            self.filled_slots = f

        if not reparse:
            return

        fp = expandpath(self._filepath)
        cfg = CMFConfigParser()
        cfg.read(fp)

        self._v_editing = 1
        # Now, for each section, find the slot
        # and generate elements for the templates
        ut = getToolByName(self, 'portal_url')
        for section in cfg.sections():
            container = self.slots.get(section)
            for option, value in cfg.items(section):
                if option != 'templates':
                    continue
                templates = map(lambda s: s.strip(), value.split(','))
                for n, template in enumerate(templates):
                    target = container.restrictedTraverse(template)
                    path = ut.getRelativeContentURL(target)
                    ob = BoundCompositeElement(str(n), path)
                    container._setObject(ob.getId(), ob)
        self._v_editing = 0

    def getTemplate(self):
        template = CompositeMixin.getTemplate(self)
        return aq_base(template).__of__(aq_parent(self))

    def __call__(self):
        """Renders the composite.
        """
        self._updateFromFS()
        return CompositeMixin.__call__(self)

    security.declareProtected(ModifyPortalContent, 'design')
    def design(self, ui=None):
        """Renders the composite with editing features.
        """
        self._updateFromFS()
        return CompositeMixin.design(self, ui=ui)

Globals.InitializeClass(FSCompositePage)

registerFileExtension('cop', FSCompositePage)
registerMetaType('Composite Page', FSCompositePage)
