# EnSimpleStaging  |  Copyright(C), 2004, Enfold Systems, LLC
# see LICENSE.txt for details

from AccessControl import ClassSecurityInfo
from Acquisition import aq_inner, aq_parent, aq_base, aq_acquire, Explicit
from BTrees.OOBTree import OOBTree
from BTrees.Length import Length
from ComputedAttribute import ComputedAttribute
from Persistence import Persistent
from webdav.WriteLockInterface import WriteLockInterface
import sets

from Products.PageTemplates.PageTemplateFile import PageTemplateFile

from Products.CMFCore.utils import UniqueObject
from Products.CMFCore.utils import getToolByName
from Products.CMFCore.utils import SimpleItemWithProperties

from Products.ZopeVersionControl.nonversioned import isAVersionableResource as isVersionable

from Products.CMFStaging.permissions import ManagePortal
from Products.CMFStaging.permissions import StageObjects
from Products.CMFStaging.staging_utils import verifyPermission
from Products.CMFStaging.staging_utils import unproxied
from Products.CMFStaging.staging_utils import getProxyReference
from Products.CMFStaging.staging_utils import cloneByPickle
from Products.CMFStaging.LockTool import LockingError

from Products.EnSimpleStaging.content.reference import InterStageReference
from Products.EnSimpleStaging.interfaces import staged
from Products.EnSimpleStaging.config import WWWDIR, logger

from Globals import InitializeClass
from Globals import DTMLFile

from OFS.ObjectManager import REPLACEABLE, NOT_REPLACEABLE

from threading import Lock

def placeholder(obj, stage, repo):
    flags = getattr(obj, '__replaceable__', NOT_REPLACEABLE)
    return flags & REPLACEABLE

def isProperChild(obj):
    # Check for objects that are returned in objectItems() from it's parent
    # but whose parent isn't even folderish (CompositePack does that).
    parent = aq_parent(obj)
    if not getattr(aq_base(parent), 'isPrincipiaFolderish', None):
        return False

    if not hasattr(obj, 'getId'):
        # A silly thing, probably acquired a view, for example the
        # '@@plone' view, iirc!
        return False

    if parent._getOb(obj.getId(), None) is None:
        return False

    # BTW, the tests above probably obviates the test below and the
    # cp_container exception in skin file
    # ess_getLastChangedPathsSince.py, 

    # XXX, sigh, we really need to write tests for all this

    # Check for nasty index_html in PloneFolder, which might
    # return an object in a wildly modified context.
    # Fixes issue #257 in Oxfam Client Support.
    ca = getattr(aq_base(aq_parent(aq_inner(obj))), obj.getId(), None)
    if isinstance(ca, ComputedAttribute):
        return False

    return obj.aq_chain == obj.aq_inner.aq_chain

# Code for protecting references across redeployment
_protectedPaths = sets.Set()

def _addProtectedPath(path):
    _protectedPaths.add(path)

def _isProtectedPath(ob_path):
    for path in _protectedPaths:
        if ob_path[:len(path)] == path:
            return True
    return False

def _protectUIDsUnder(obj):
    path = obj.getPhysicalPath()
    _protectedPaths.add(path)

def _unprotectUIDsUnder(obj):
    path = obj.getPhysicalPath()
    _protectedPaths.remove(path)


class StagingError (Exception):
    """Error while attempting to stage an object"""

class VersionPathInfo(Persistent):

    def __init__(self, id='', info=None):
        self.id = id
        self.info = info
        self.child = OOBTree()

    def __getitem__(self, key):
        if not self.child.has_key(key):
            self[key] = VersionPathInfo(key)
        return self.child[key]

    def __setitem__(self, key, value):
        self.child[key] = value

    def items(self):
        return self.child.items()

    def keys(self):
        return self.child.keys()

    def values(self):
        return self.child.values()

    def traverse(self, path=None):
        if not path:
            return self
        parts = path.split('/')
        parts = filter(None, parts)
        obj = self
        for part in parts:
            obj = obj[part]
        return obj

    def setInfo(self, obj):
        self.info = obj.__vc_info__.clone()

class SourceStageInfo(Explicit, Persistent):
    """ XXX This class is here solely for backward compatibility
    We should provide a migration that removes instances of this
    object from deployed objects.
    """

    __allow_access_to_unprotected_subobjects__ = 1

    def __init__(self, obj, source_stage):
        self.source_stage = source_stage

    def getObject(self):
        return self.aq_inner.aq_parent

    def getWorkspacesTool(self):
        return getToolByName(self.getObject(), 'portal_workspaces')

    def getSourceStage(self):
        st = self.getWorkspacesTool()
        return st.getStageByName(self.source_stage)

    def getSourceStageName(self):
        return self.source_stage

    def getSourceObject(self):
        st = self.getWorkspacesTool()
        objPath = st.getRelativeStageURL(self.getObject())
        sourceStage = self.getSourceStage()
        return sourceStage.restrictedTraverse(objPath)

class WorkspacesTool(UniqueObject, SimpleItemWithProperties):
    """ A tool that maps EnSimpleStaging workspaces to CMFStaging like
    stages
    """
    id = "portal_workspaces"
    meta_type = 'Workspaces Tool'
    security = ClassSecurityInfo()

    _stage_info = None

    __implements__ = (WriteLockInterface,)

    manage_options = (
        {'label': 'Overview', 'action': 'manage_overview'},
        {'label': 'Stages', 'action': 'manage_stagesForm'},
        ) + SimpleItemWithProperties.manage_options

    repository_name = 'VersionRepository'

    _properties = (
        {'id': 'repository_name', 'type': 'string', 'mode': 'w',
         'label': 'ID of the version repository'},
        )

    security.declareProtected(ManagePortal, 'manage_overview' )
    manage_overview = DTMLFile('workspacesToolOverview', WWWDIR)

    security.declareProtected(ManagePortal, 'manage_stagesForm')
    manage_stagesForm = PageTemplateFile('stages', WWWDIR,
                                         __name__='manage_stagesForm')

    def _orderStageItems(self, items):
        # do a little schwartz transform to reorder the stages.
        # we need them to be in reverse path-lexicografical order
        # so that the more specific (longer) paths match first
        items = [(stagePath.split('/'), (stageId, stageTitle, stagePath))
                  for stageId, stageTitle, stagePath in items]
        items.sort()
        items.reverse()
        items = [ stageInfo for stagePath, stageInfo in items ]
        return items

    security.declareProtected(ManagePortal, 'getWorkspaces')
    def getWorkspaces(self):
        pc = getToolByName(self, 'portal_catalog')
        return [brain.getObject()
                for brain in pc(portal_type='StagingArea')]

    security.declareProtected(ManagePortal, 'getStageItems')
    def getStageItems(self):
        """Returns the stage declarations (for the management UI.)
        """
        stages = []
        for workspace in self.getWorkspaces():
            ssi = workspace.getSourceStageInfo()
            if ssi:
                stages.append(ssi)
            dsi = workspace.getDestinationStageInfo()
            if dsi:
                stages.append(dsi)
        stages = self._orderStageItems(stages)
        return stages

    def __init__(self, *args, **kw):
        self._stage_info = OOBTree()

    def _verifyObjStage(self, obj, stage, repo):
        if obj is not None:
            # if the object is there just to be replaced
            # we pretend it's not there
            if placeholder(obj, stage, repo):
                return None
            # if the object is not really a subobject of it's parent
            # rip it out. This should be the actual test CMFStaging
            # makes :-)
            # testing only the last segment should be enough, as this
            # test will be applied recursively for all objects to be
            # published
            if not isProperChild(obj):
                return None
        return obj

    security.declarePublic('getStagesInfo')
    def getStagesInfo(self):
        stages = {}
        for name, title, path in self.getStageItems():
            stages[path] = (name, title)
        return stages

    def getStageByName(self, stage_name):
        portal = aq_parent(aq_inner(self))
        for name, title, path in self.getStageItems():
            if name == stage_name:
                return portal.restrictedTraverse(path, None)
        return None

    def _recurseAndApply(self, obj, apply_func):
        """ quick and hacked up substitute for portal_catalog.ZopeFindAndApply(). Needs tests """
        for item in obj.objectValues():
            if apply_func(item) and getattr(aq_base(item), 'isPrincipiaFolderish', None):
                self._recurseAndApply(item, apply_func)

    def recursiveWipeStage(self, stage_name, hardWipe=False):
        """Try to wipe out objects that are under
        version control. If a folder is not empty, don't
        remove it.
        """
        stage = self.getStageByName(stage_name)
        assert stage is not None, ('Invalid stage name or stage points to a '
                                   'non-existing destination: %s' % stage_name)
        vt = getToolByName(self, 'portal_versions')
        self._recursiveWipe(stage, vt, hardWipe)
        return 'foo'

    def _recursiveWipe(self, obj, vt=None, hardWipe=False):
        if vt is None:
            vt = getToolByName(self, 'portal_versions')
        if hardWipe:
            # just wipe out all of the destination
            ids = obj.objectIds()
        else:
            ids = []
            obj_stage = self.getStageFor(obj)
            for item in obj.objectValues():
                if self.getStageFor(item) != obj_stage:
                    # item is under control of a different stage we can't
                    # trash it or it's contents
                    continue
                if item.aq_explicit.isPrincipiaFolderish:
                    self._recursiveWipe(item, vt)
                    if item.objectIds():
                        continue
                if not vt.isUnderVersionControl(item):
                    # on a hardWipe, we wipe objects even if not under vc
                    continue
                ids.append(item.getId())
        obj.manage_delObjects(ids=ids)

    def checkInObjects(self, objects, message=None, vt=None):
        if vt is None:
            vt = getToolByName(self, 'portal_versions')
        for ob in objects:
            vt.maybeCheckout(ob, message)
            vt.maybeCheckin(ob, message)

    security.declareProtected(ManagePortal, 'recursiveTag')
    def recursiveTag(self, obj, from_stage,
                     label, message=None,
                     vt=None, wt=None, force_tag=False):
        # This is used for tagging in 'in place' staging. We need to
        # check for workflow states to be deployed, just like in
        # recursivePublish.
        if vt is None:
            vt = getToolByName(self, 'portal_versions')
        if wt is None:
            wt = getToolByName(self, 'portal_workflow')
        source = self.getStageByName(from_stage)
        states = source.getDeployStates()
        if message is None:
            message = 'Auto-saved'
        self._recursiveTag(
            obj=obj, from_stage=from_stage, label=label,
            message=message, vt=vt, wt=wt, states=states,
            force_tag=force_tag)

    def _recursiveTag(self, obj, from_stage, label, message,
                      vt, wt=None, states=(), force_tag=False):
        # function for _recurseAndApply
        def _doTag(item, states=states):
            if not isVersionable(item):
                # we can't checkin non-versionables
                return
            if item.aq_base.__class__ == InterStageReference:
                # we should skip stage references
                # the test above could be improved a lot
                # say, with an interface check
                return
            # If we got a list of states and a workflow tool instance,
            # check that the object is in one of the valid states
            # before proceeding.
            if states and wt is not None:
                states = tuple(states) + (None,)
                # XXX Get rid of hardcoded 'review_state' here.
                state = wt.getInfoFor(item, 'review_state', default=None)
                if state not in states:
                    # Not in a state marked for deployment.
                    return
            # if 'force_tag' is set, we do a vt.maybeCheckout() so
            # that we force a new revision of the object.
            if force_tag:
                vt.maybeCheckout(item, message)
            # we do a vt.maybeCheckin() here. It will create and store
            # another version for objects that are checked out, even
            # if they weren't changed. this should only happen once
            # for objects left checked-out from previous versions of
            # ESS, as new versions of ESS leave objects checked-in by
            # default, and only check them out when a new version is
            # actually needed.
            vt.maybeCheckin(item, message)
            # no need to tag, we keep versions for ourselves
            #vt.tag(item, label)
            self.annotateTag(item, label, from_stage)
            return True

        # first we setup the root of the VersionPath tree
        self._setupVersionPathInfoRoot(label, from_stage)
        # now we dive
        self._recurseAndApply(obj, apply_func=_doTag)

    security.declarePublic('recursivePublish')
    def recursivePublish(self, obj, from_stage, to_stages,
                         label=None, message=None, vt=None,
                         wt=None, wipe=True, hardWipe=False):
        # This method is used by full publication, so we have to pass
        # on a workflow tool and the states that should be deployed.
        if vt is None:
            vt = getToolByName(self, 'portal_versions')
        if wt is None:
            wt = getToolByName(self, 'portal_workflow')
        source = self.getStageByName(from_stage)
        states = source.getDeployStates()
        # Prepare the code to call while locked.
        # Need to pass message as parameter to avoid an
        # UnboundLocalError
        def recursivePublish(message=message):
            if wipe:
                # wiping stages wrecks AT reference engine if we
                # don't take some protective steps.
                _protectUIDsUnder(obj)
                try:
                    for stage in to_stages:
                        self.recursiveWipeStage(stage, hardWipe=hardWipe)
                finally:
                    _unprotectUIDsUnder(obj)
            if message is None:
                message = 'Auto-saved'
            self._recursivePublish(
                obj=obj, from_stage=from_stage, to_stages=to_stages,
                label=label, message=message, vt=vt, wt=wt, states=states)
        # call the prepared code while locked
        self._publicationLockedCall(recursivePublish)
        return

    def _isProtectedUID(self, uid):
        # we need the main UID catalog for this (not the one in the
        # subsite)
        portal = getToolByName(self, 'portal_url').getPortalObject()
        uc = getToolByName(portal, 'uid_catalog')
        relpath = uc(UID=uid)[0].getPath()
        # .getPath() from the uid_catalog returns a relative path
        path = portal.getPhysicalPath() + tuple(relpath.split('/'))
        # check the path is inside one of the protected paths
        return _isProtectedPath(path)

    def _recursivePublish(self, obj, from_stage, to_stages,
                          label, message, vt, wt, states):
        # function for self._recurseAndApply
        def doPublish(item):
            return self._doPublish(
                item, to_stages, label, message,
                vt, wt, states, full=True)
        # first we setup the root of the VersionPath tree
        self._setupVersionPathInfoRoot(label, from_stage)
        # now we dive
        self._recurseAndApply(obj, apply_func=doPublish)

    def publishObjects(self, objs, to_stages,
                       message=None, vt=None):
        # This method is used by partial publication, most checks
        # should have already be performed, like checking workflow
        # states, etc.
        if vt is None:
            vt = getToolByName(self, 'portal_versions')
        # prepare the code to call while locked
        def publishObjects():
            for obj in objs:
                # don't tag the objects (label -> None)
                self._doPublish(obj, to_stages, None,
                                message, vt, full=False)
        self._publicationLockedCall(publishObjects)

    def _doPublish(self, item, to_stages, label=None,
                   message=None, vt=None, wt=None,
                   states=(), full=True):
        __traceback_info__ = (item, to_stages)
        if not isVersionable(item):
            # we can't checkin non-versionables
            return
        if not isProperChild(item):
            # only proper subobjects of their containers can be
            # staged
            return
        if item.aq_base.__class__ == InterStageReference:
            # We should skip stage references.
            # The test above could be improved a lot
            # say, with an interface check
            return
        if states and wt is not None:
            states = tuple(states) + (None,)
            # XXX Get rid of hardcoded 'review_state' here.
            state = wt.getInfoFor(item, 'review_state', default=None)
            if state not in states:
                # Not in a state marked for deployment.
                return
        if label is not None:
            vt.maybeCheckout(item, message)
            vt.maybeCheckin(item, message)
            self.annotateTag(item, label)
        self.updateStages2(item, to_stages, 'Updated by recursive publication')
        self._notifyObjectStaged(item, to_stages, full=full)
        return True

    def _notifyObjectStaged(self, item, to_stages, full):
        # find each destination object and trigger an
        # ObjectStagedEvent notification
        from_stage = self.getStageOf(item)
        item_physical_path = '/'.join(item.getPhysicalPath())
        if from_stage is None:
            raise StagingError("Object %s is not in any stage" %
                               item_physical_path)
        for to_stage in to_stages:
            to_stage_obj = self.getStageByName(to_stage)
            itemPath = self.getRelativeStageURL(item)
            dstItem = to_stage_obj.unrestrictedTraverse(itemPath)
            logger.debug('Staged %s (full=%s)' % (item_physical_path, full))
            staged(item, dstItem, full)

    def recursiveRevertToLabel(self, obj, label):
        vt = getToolByName(self, 'portal_versions')
        p_info = self.getTagAnnotationFor(obj, label)
        p_info = filter(lambda i: i[1].info is not None, p_info)
        p_info = dict(p_info)
        for item in obj.objectValues():
            # decide if the subobject can be reverted or not
            i_info = p_info.get(item.getId())
            removeIt = False
            if i_info is None:
                # item didn't exist back then
                removeIt = True
            elif not vt.isUnderVersionControl(item):
                # Item isn't vc controlled yet, so it's a diferent object
                removeIt = True
            elif vt.getHistoryId(item) != i_info.info.history_id:
                # objects belong to different histories
                removeIt = True
            if removeIt:
                # object needs to be deleted or fetched anew.
                obj._delObject(item.getId())
                continue
            # this below is not needed, vt does it's own 'maybeCheckin
            #vt.maybeCheckin(item, 'Auto-saved')
            # Item existed, still exists and belongs to the same history,
            # use normal means
            #vt.revertToVersion(item, label)
            vt.revertToVersion(item, i_info.info.version_id)
            # reverting the object will usually stick it.
            vt.maybeUnstickObject(item)
            # force recataloging since we didn't use _setObject
            if getattr(aq_base(item), 'reindexObject', False):
                item.reindexObject()
            if isVersionable(item) and item.aq_explicit.isPrincipiaFolderish:
                self.recursiveRevertToLabel(item, label)
            # make sure we don't try to check out the object again below
            del p_info[item.getId()]
        for id, i_info in p_info.items():
            # checkout objects that weren't updated above
            # but existed when the label was made
            new_ob = vt.getObjByInfo(i_info.info)
            obj._setObject(id, new_ob, set_owner=0)
            new_ob = obj._getOb(id)
            self.recursiveRevertToLabel(new_ob, label)
        return

    def _getStage(self, portal, path):
        if not path or path == ".":
            return portal
        else:
            return portal.restrictedTraverse(path, None)

    security.declarePublic('getStageFor')
    def getStageFor(self, obj):
        """Returns the stage the object is in the context of.
        """
        # this assumes self.getStageItems() is in an innermost to outermost
        # order
        portal = aq_parent(aq_inner(self))
        for stage_name, stage_title, path in self.getStageItems():
            stage = self._getStage(portal, path)
            if stage is not None and obj.aq_inContextOf(stage, 1):
                return stage
        return None

    security.declarePublic('getStageOf')
    def getStageOf(self, obj):
        """Returns the stage name the object is in the context of.
        """
        portal = aq_parent(aq_inner(self))
        for stage_name, stage_title, path in self.getStageItems():
            stage = self._getStage(portal, path)
            if stage is not None and obj.aq_inContextOf(stage, 1):
                return stage_name
        return None

    security.declarePublic('getRelativeStageURL')
    def getRelativeStageURL(self, obj):
        ut = getToolByName(self, 'portal_url')
        stage = self.getStageFor(obj)
        path = ut.getRelativeContentPath(stage)
        obj_path = ut.getRelativeContentPath(obj)
        return '/'.join(obj_path[len(path):])

    def _recursiveGetPath2History(self, p_info, basepath, results):
        for id, sub_p_info in p_info.items():
            if sub_p_info.info is None:
                # No information for this path.
                continue
            path = basepath + (id,)
            results.append((path, sub_p_info.info.history_id))
            self._recursiveGetPath2History(sub_p_info, path, results)

    def getPath2HistoryIdMap(self, stage, label):
        ti = self._stage_info[stage]
        p_info = ti[label]
        results = []
        basepath = ()
        self._recursiveGetPath2History(p_info, basepath, results)
        results.sort() # sort by reverse lexicographical path order
        return results

    def _recursiveGetPath2VersionInfoMap(self, p_info, basepath, results):
        for id, sub_p_info in p_info.items():
            if sub_p_info.info is None:
                # No information for this path.
                continue
            path = basepath + (id,)
            results.append((path,
                            sub_p_info.info.history_id,
                            sub_p_info.info.version_id))
            self._recursiveGetPath2VersionInfoMap(sub_p_info, path, results)

    def getPath2VersionInfoMap(self, stage, label):
        ti = self._stage_info[stage]
        p_info = ti[label]
        results = []
        basepath = ()
        self._recursiveGetPath2VersionInfoMap(p_info, basepath, results)
        results.sort() # sort by reverse lexicographical path order
        return results

    def _setupVersionPathInfoRoot(self, label, stage):
        ti = self._stage_info.get(stage)
        if ti is None:
            self._stage_info[stage] = ti = OOBTree()
        p_info = ti.get(label)
        if p_info is None:
            ti[label] = VersionPathInfo()

    def annotateTag(self, obj, label, stage=None):
        if stage is None:
            stage = self.getStageOf(obj)
        ti = self._stage_info.get(stage)
        p_info = ti.get(label)
        rel_path = self.getRelativeStageURL(obj)
        item = p_info.traverse(rel_path)
        item.setInfo(obj)

    def isValidLabel(self, stageName, label):
        return bool(self._stage_info.get(stageName) and
                    self._stage_info[stageName].get(label))

    def getTagAnnotationFor(self, obj, label, stage=None):
        rel_path = self.getRelativeStageURL(obj)
        if stage is None:
            stage = self.getStageOf(obj)
        return self.getTagAnnotationForPath(path=rel_path,
                                            label=label,
                                            stage=stage)

    def getVersionInfoForPath(self, path, label, stage):
        ti = self._stage_info.get(stage)
        if ti is None:
            return
        p_info = ti.get(label)
        if p_info is None:
            ti[label] = p_info = VersionPathInfo()
        return p_info.traverse(path)

    def getTagAnnotationForPath(self, path, label, stage):
        p_info = self.getVersionInfoForPath(path, label, stage)
        if p_info is None:
            return
        return p_info.items()

    security.declarePublic('updateStages')
    def updateStages(self, obj, from_stage, to_stages, message=''):
        """Backward compatibility wrapper.

        Calls updateStages2().  Note that the source stage is
        specified twice, first in the context of obj, second in
        from_stage.  updateStages2() eliminates the potential
        ambiguity by eliminating from_stage.
        """
        s = self.getStageOf(obj)
        if s != from_stage:
            raise StagingError("Ambiguous source stage")
        self.updateStages2(obj, to_stages, message)


    security.declarePublic('updateStages2')
    def updateStages2(self, obj, to_stages, message=''):
        """Updates objects to match the version in the source stage.
        """
        verifyPermission(StageObjects, obj)
        from_stage = self.getStageOf(obj)
        if from_stage is None:
            raise StagingError("Object %s is not in any stage" %
                               '/'.join(obj.getPhysicalPath()))
        if from_stage in to_stages or not to_stages:
            raise StagingError("Invalid to_stages parameter, %s" % str(to_stages))

        if aq_base(unproxied(obj)) is not aq_base(obj):

            ref = obj.__dict__.get("_Proxy__reference")

            if ref is None:
                # Carefully wrap an *un*proxied version of obj in the same
                # context:
                IAW = ImplicitAcquisitionWrapper
                obj = IAW(unproxied(obj), aq_parent(aq_inner(obj)))
                proxy = None
            else:
                # obj is a proxy.  Find the wrapped target and update that
                # instead of the reference.  Note that the reference will
                # be updated with the container.
                proxy = obj
                obj = ref.getTarget(obj)
                # Decide whether the reference should be staged at the
                # same time.  If the reference is contained in a
                # non-versioned container, the reference should be staged.
                # OTOH, if the reference is in a versioned container,
                # staging the container will create the reference, so the
                # reference should not be staged by this operation.
                repo = self._getVersionRepository()
                if repo.isUnderVersionControl(aq_parent(aq_inner(proxy))):
                    proxy = None
        else:
            proxy = None

        # Check containers first.
        cmap = self._getObjectStages(obj, get_container=1)
        self._checkContainers(obj, to_stages, cmap)
        proxy_cmap = None
        if proxy is not None:
            # Check the containers of the reference also.
            proxy_cmap = self._getObjectStages(proxy, get_container=1)
            self._checkContainers(proxy, to_stages, proxy_cmap)

        self._updateObjectStates(obj, cmap, to_stages)
        if proxy is not None:
            # Create and update the reference objects also.
            self._updateReferences(proxy, proxy_cmap, to_stages)


    def _updateObjectStates(self, source_object, container_map, to_stages):
        """Internal: updates the state of an object in specified stages.

        Uses version control to do the propagation.
        """
        repo = self._getVersionRepository()
        object_map = self._getObjectStages(source_object)
        version_info = repo.getVersionInfo(source_object)
        version_id = version_info.version_id
        history_id = version_info.history_id

        # Update and/or copy the object to the different stages.
        for stage_name, ob in object_map.items():
            if stage_name in to_stages:
                container = container_map[stage_name]
                source_id = source_object.getId()
                if ob is None:
                    # The object has not yet been created in the stage.
                    # Copy from the repository to the given stage.
                    ob = repo.getVersionOfResource(history_id, version_id)
                    container = container_map[stage_name]
                    # Make a copy and put it in the new place.
                    container._setObject(source_id, ob, set_owner=0)
                    # the line above will trigger a recataloging on
                    # catalogAware objects
                else:
                    if not repo.isUnderVersionControl(ob):
                        p = '/'.join(ob.getPhysicalPath())
                        raise StagingError(
                            'The object "%s", not under version control, '
                            'is in the way.' % p)
                    if repo.getVersionInfo(ob).history_id != history_id:
                        p = '/'.join(ob.getPhysicalPath())
                        p2 = '/'.join(source_object.getPhysicalPath())
                        raise StagingError(
                            'The object "%s", backed by a different '
                            'version history than "%s", '
                            'is in the way.' % (p, p2))
                    # the line below doesn't trigger a catalog update
                    repo.updateResource(unproxied(ob), version_id)
                    # so we trigger it manually
                    if getattr(aq_base(ob), 'reindexObject', False):
                        ob.reindexObject()
                assert source_id == ob.getId(), "source and destination object IDs don't match"
                assert ( aq_base(container._getOb(source_id)) is
                         aq_base(ob) ), "destination object is not under a proper id inside the container"

    def _updateReferences(self, proxy, container_map, to_stages):
        """Internal: creates and updates references.
        """
        # Note that version control is not used when staging
        # reference objects.
        ref = getProxyReference(proxy)
        object_map = self._getObjectStages(proxy)
        ref_id = ref.getId()
        for stage_name, ob in object_map.items():
            if stage_name in to_stages:
                if ob is not None:
                    # There is an object at the reference target.
                    if type(aq_base(ob)) is not type(aq_base(proxy)):
                        p = '/'.join(ob.getPhysicalPath())
                        raise StagingError(
                            'The object "%s", which is not a reference, '
                            'is in the way.' % p)
                    # Delete the reference.
                    container = container_map[stage_name]
                    container._delObject(ref_id)

                # Copy the reference from the source stage.
                container = container_map.get(stage_name, None)
                if container is None:
                    # This can happen if a site doesn't yet exist on
                    # the stage.
                    p = '/'.join(proxy.getPhysicalPath())
                    raise StagingError(
                        'The container for "%s" does not exist on "%s"'
                        % (p, stage_name))
                # Duplicate the reference.
                ob = cloneByPickle(aq_base(ref))
                container._setObject(ob.getId(), ob, set_owner=0)


    security.declarePublic('removeStages')
    def removeStages(self, obj, stages):
        """Removes the copies on the given stages.
        """
        # If the object is a reference or proxy, this removes only the
        # reference or proxy; this is probably the right thing to do.
        verifyPermission(StageObjects, obj)
        object_map = self._getObjectStages(obj)
        container_map = self._getObjectStages(obj, get_container=1)
        id = obj.getId()
        for stage_name, container in container_map.items():
            if object_map.get(stage_name) is not None:
                if container is not None and stage_name in stages:
                    container._delObject(id)


    security.declarePublic('getVersionIds')
    def getVersionIds(self, obj, include_status=0):
        """Retrieves object version identifiers in the different stages.
        """
        verifyPermission(StageObjects, obj)
        return self._getObjectVersionIds(obj, include_status)


    def _checkContainers(self, obj, stages, containers):
        for stage in stages:
            if containers.get(stage) is None:
                p = '/'.join(obj.getPhysicalPath())
                raise StagingError(
                    'The container for "%s" does not exist on "%s"'
                    % (p, stage))


    security.declarePublic('stageExists')
    def stageExists(self, stage):
        """Return 1 if named stage exists, else 0."""
        portal = aq_parent(aq_inner(self))
        for stage_name, stage_title, stage_path in self.getStageItems():
            if stage_name == stage:
                stage = self._getStage(portal, stage_path)
                if stage: return 1
                else: return 0
        return 0

    security.declarePublic('checkContainers')
    def checkContainers(self, obj, stages):
        """Verifies that the container exists on the given stages.

        If the container is missing on one of the stages, an exception
        is raised.
        """
        verifyPermission(StageObjects, obj)
        containers = self._getObjectStages(obj, get_container=1)
        self._checkContainers(obj, stages, containers)
        return 1

    def _getObjectStages(self, obj, get_container=0):
        """Returns a mapping from stage name to object in that stage.

        Objects not in a stage are represented as None.
        """
        portal = aq_parent(aq_inner(self))
        stages = {}
        rel_path = None
        ob_path = obj.getPhysicalPath()
        for stage_name, stage_title, path in self.getStageItems():
            stage = self._getStage(portal, path)
            stages[stage_name] = stage
            if stage is not None and obj.aq_inContextOf(stage, 1):
                if rel_path is not None:
                    # Can't tell what stage the object is in!
                    raise StagingError("The stages overlap")
                # The object is from this stage.
                stage_path = stage.getPhysicalPath()
                assert ob_path[:len(stage_path)] == stage_path
                rel_path = ob_path[len(stage_path):]

        if rel_path is None:
            raise StagingError("Object %s is not in any stage" % (
                '/'.join(ob_path)))

        if get_container:
            # Get the container of the object instead of the object itself.
            # To do that, we just traverse to one less path element.
            rel_path = rel_path[:-1]

        res = {}
        repo = self._getVersionRepository()
        for stage_name, stage_title, path in self.getStageItems():
            stage = stages[stage_name]
            if stage is not None:
                obj = stage.restrictedTraverse(rel_path, None)
                obj = self._verifyObjStage(obj, stage, repo)
            else:
                obj = None
            res[stage_name] = obj
        return res

    def _getObjectVersionIds(self, obj, include_status=0):
        repo = self._getVersionRepository()
        stages = self._getObjectStages(obj)
        res = {}
        for stage_name, obj in stages.items():
            u_obj = unproxied(obj)
            if obj is None or not repo.isUnderVersionControl(u_obj):
                res[stage_name] = None
            else:
                info = repo.getVersionInfo(u_obj)
                v = info.version_id
                if include_status and info.status == info.CHECKED_OUT:
                    v = str(v) + '+'
                res[stage_name] = v
        return res

    security.declarePublic('isStageable')
    def isStageable(self, obj):
        """Returns a true value if the object can be staged."""
        verifyPermission(StageObjects, obj)
        repo = self._getVersionRepository()
        if not repo.isAVersionableResource(unproxied(obj)):
            return 0
        if not getattr(obj, '_stageable', 1):
            return 0
        # An object is stageable only if it is located in one of the stages.
        portal = aq_parent(aq_inner(self))
        for stage_name, stage_title, path in self.getStageItems():
            stage = self._getStage(portal, path)
            if stage is not None and obj.aq_inContextOf(stage, 1):
                return 1
        return 0

    def _getVersionRepository(self):
        repo = aq_acquire(self, self.repository_name, containment=1)
        return repo

    security.declarePublic('isStageable')
    def checkDuplicateStages(self):
        """Raises an error if there are 2 stages with the same path"""
        stagePaths = {}
        for stageId, stageTitle, stagePath in self.getStageItems():
            if stagePath in stagePaths:
                raise ValueError(
                    "stage at '%s' is duplicated for '%s' (%s) and '%s' (%s)" %
                    (stagePath,
                     stageId, stageTitle,
                     stagePaths[stagePath][0],
                     stagePaths[stagePath][1]))
            stagePaths[stagePath] = (stageId, stageTitle)

    def _publicationLockedCall(self, callable, *args, **kw):
        """returns this instance lock"""
        lt = getToolByName(self, 'portal_lock')
        try:
            lt.lock(self)
        except LockingError:
            raise StagingError('Workspaces are locked, this means a '
                               'workspace deployment is ongoing. '
                               'Please try again later')
        try:
            result = callable(*args, **kw)
        finally:
            lt.unlock(self)
        return result


InitializeClass(WorkspacesTool)
