# EnSimpleStaging  |  Copyright(C), 2004, Enfold Systems, LLC
# see LICENSE.txt for details

import os, sys
from sets import Set

if __name__ == '__main__':
    execfile(os.path.join(sys.path[0], 'framework.py'))

from Testing import ZopeTestCase
from Testing.ZopeTestCase.functional import Functional
from Products.PloneTestCase import PloneTestCase

for product in ('Archetypes',
                'PortalTransforms',
                'ZopeVersionControl',
                'CMFStaging',
                'EnSimpleStaging',
                'MimetypesRegistry',
#                'SecureMailHost',
                ):
    PloneTestCase.installProduct(product)
del product

from cStringIO import StringIO
from Products.CMFCore.utils import getToolByName
from Products.Archetypes.tests.utils import makeContent
from Products.EnSimpleStaging.tests.base import TestStaging

PloneTestCase.setupPloneSite()

class TestPublishing(TestStaging):

    def testStagePublishing(self):
        self.setupStagingAreas()
        mkt = self.home.mkt
        adv = self.home.adv
        public = self.public
        public.invokeFactory('Folder', id='marketing')
        public.invokeFactory('Folder', id='advertising')
        # Create index_html on public_website
        # to see if we trigger a bug in staging.
        public.invokeFactory('Document', id='index_html')

        mkt.setStagePath('public_website/marketing')
        adv.setStagePath('public_website/advertising')

        self.failUnless(hasattr(public, 'marketing'))
        self.failUnless(hasattr(public, 'advertising'))

        self._populateFolder(mkt)
        self._populateFolder(adv)

        self.failIf(self._checkStructure(mkt))
        self.failIf(self._checkStructure(adv))
        self.failUnless(self._checkStructure(public.marketing))
        self.failUnless(self._checkStructure(public.advertising))

        # Check for relative stage URL
        rcu = self.wt.getRelativeStageURL
        self.assertEquals(rcu(mkt), '')
        self.assertEquals(rcu(mkt.doc1), 'doc1')
        self.assertEquals(rcu(mkt.folder1), 'folder1')
        self.assertEquals(rcu(mkt.folder2.folder22.folder221),
                          'folder2/folder22/folder221')
        mkt.publishStage()
        adv.publishStage()

        errors = self._checkStructure(public.marketing)
        self.failIf(errors, errors)
        errors = self._checkStructure(public.marketing.advertising)
        self.failIf(errors, errors)

    def testAfterInitialPublishing(self):
        # Add some content after initially publishing a stage
        self.testStagePublishing()
        # Now we have some Stages already setup and under version
        # control.  Now lets add some folders and documents in
        # the staging area and try to publish it.
        mkt_folder11 = self.home.mkt.folder1.folder11
        mkt_folder11.invokeFactory('Folder', id='folder111')
        folder111 = mkt_folder11.folder111
        folder111.invokeFactory('Document', id='index_html')
        self.home.mkt.publishStage()
        public = self.public
        pf11 = public.marketing.folder1.folder11
        self.failUnless(hasattr(pf11, 'folder111'))
        self.failUnless(hasattr(pf11.folder111, 'index_html'))

    def testNonExistingTarget(self):
        self.setupStagingAreas()
        mkt = self.home.mkt
        adv = self.home.adv
        public = self.public
        # Create index_html on public_website
        # to see if we trigger a bug in staging.
        public.invokeFactory('Document', id='index_html')

        # Note public_website/marketing doesn't exist.
        self.failIf('marketing' in public.objectIds())
        mkt.setStagePath('public_website/marketing')
        mkt.invokeFactory('Document', id='test_doc')

        # Should give an assertion error
        self.assertRaises(AssertionError, mkt.publishStage)


class TestPartialDeployment(TestStaging):

    def _checkLabelsIntegrity(self, workspace):
        labelset = Set()
        for annotation in workspace.getAnnotations():
            label = annotation['label']
            if label in labelset:
                self.fail('label %s was present more than once' % label)
            labelset.add(label)

    def test_getPath2HistoryIdMap(self):
        self.setupStagingAreas()
        mkt = self.home.mkt
        mkt.setStagePath('public_website')
        pubnode = mkt
        # let's add some content and publish
        pubnode.invokeFactory('Folder', id='f1')
        pubnode.f1.invokeFactory('Document', id='d1')
        pubnode.invokeFactory('Folder', id='f2')
        mkt.publishStage('test1')
        stage = self.wt.getStageOf(mkt)
        pathmap = self.wt.getPath2HistoryIdMap(stage,
                                               mkt._globalLabel('test1'))
        rightmap = [(('f1',), pubnode.f1.__vc_info__.history_id),
                    (('f1','d1'), pubnode.f1.d1.__vc_info__.history_id),
                    (('f2',), pubnode.f2.__vc_info__.history_id),]
        self.assertEquals(pathmap, rightmap)

    def test_interpretOpcodes(self):
        seq1 = 'a1 aa1 ab1 b1 ba1 bb1'.split()
        seq2 = 'a1 ab1 b1 ba2 bb2 c'.split()
        from Products.EnSimpleStaging.content.stage import interpretOpcodes
        from difflib import SequenceMatcher
        opcodes = SequenceMatcher(None, seq1, seq2).get_opcodes()
        removes, adds = interpretOpcodes(opcodes, seq1, seq2)
        self.assertEquals(removes, 'aa1 ba1 bb1'.split())
        self.assertEquals(adds, 'ba2 bb2 c'.split())

    def test_mayPublishIncrementally(self):
        self.setupStagingAreas()
        mkt = self.home.mkt
        mkt.setStagePath('public_website')
        pubnode = mkt
        # incremental publication should be disabled
        self.assertEquals(mkt.mayPublishIncrementally(), False)
        # let's add some content and publish
        pubnode.invokeFactory('Folder', id='f1')
        pubnode.invokeFactory('Folder', id='f2')
        pubnode.invokeFactory('Document', id='d1')
        pubnode.invokeFactory('Document', id='d4')
        mkt.publishStage('test1')
        # should allow publishing now
        self.assertEquals(mkt.mayPublishIncrementally(), True)

    def test_partial_deployment(self):
        self.setupStagingAreas()
        mkt = self.home.mkt
        mkt.setStagePath('public_website')
        pubnode = mkt
        # let's add some content and publish
        pubnode.invokeFactory('Folder', id='f1')
        pubnode.invokeFactory('Folder', id='f2')
        pubnode.invokeFactory('Document', id='d1')
        pubnode.invokeFactory('Document', id='d4')
        mkt.publishStage('test1')
        # let's sleep a little
        import time
        time.sleep(1)
        pubnode.f2.edit(title='folder 2')
        pubnode.invokeFactory('Document', id='d2')
        pubnode.update(title='foo') # just to make sure it's not deployed
        d1relPath = self.wt.getRelativeStageURL(pubnode.d1) # for later
        self.subcommit()
        pubnode.manage_renameObject('d1', 'd3')
        pubnode.manage_delObjects(ids=['d4'])
        pubnode.manage_clone(pubnode.d3, 'd4')
        import time
        time.sleep(1)
        # the above should cause some detectable modifications, right?
        lastAnnotDate = mkt.getLastAnnotation()['date']
        lastChanged = [mkt.restrictedTraverse(path)
                       for path in mkt.getLastChangedPaths()]
        self.failUnless(pubnode.modified() >
                        lastAnnotDate,
                        "pub. node modification wasn't registered: (%s, %s)" %
                        (pubnode.modified(),
                         lastAnnotDate))
        self.failIf(pubnode in lastChanged,
                    "pub. node was set to be published")
        self.failUnless(pubnode.f2.modified() >
                        lastAnnotDate,
                        "modification wasn't registered: (%s, %s)" %
                        (pubnode.f2.modified(),
                         lastAnnotDate))
        self.failUnless(pubnode.f2 in lastChanged,
                        "modification wasn't noticed")
        self.failUnless(pubnode.d2 in lastChanged,
                        "addition wasn't noticed")
        d3relPath = self.wt.getRelativeStageURL(pubnode.d3)
        d4relPath = self.wt.getRelativeStageURL(pubnode.d4)
        mkt.saveStage('save1') # temporary save to let getLastMovements() work
        remPaths, addPaths = mkt.getLastMovements()
        # d1 was moved to d3
        self.failUnless(d1relPath in remPaths)
        self.failUnless(d3relPath in addPaths)
        # d4 was removed and d3 was copied in it's place
        self.failUnless(d4relPath in remPaths)
        self.failUnless(d4relPath in addPaths)
        # let's pretend the last tagging didn't happen
        mkt._si_length.change(-1)
        # that being settled, we need to see if it's publishing the stuff
        # but let's get the suggested next label first
        nl1 = int(mkt.getNextLabel())
        # now we publish
        mkt.publishLastChanged('test2')
        # and check that labels are behaving correctly
        nl2 = int(mkt.getNextLabel())
        self.assertEquals(nl2, nl1 + 1)
        self._checkLabelsIntegrity(mkt)
        # let's check the deployed objects
        self.assertEquals(self.public.f2.Title(), 'folder 2')
        self.failIf('d1' in self.public.objectIds())
        self.failUnless('d3' in self.public.objectIds())
        # and see that the reported number of affected objects is correct
        annot = mkt.getLastAnnotation()
        affected = annot['count']
        self.assertNotEquals(mkt.count(), affected)
        self.assertEquals(affected, 4)
        # and that this last operation can be reverted to:
        self.failUnless(mkt.isReversibleAnnotation(annot))

    def _checkVersions(self, obj, numVersions, numLabels=None):
        if numLabels is None:
            numLabels = numVersions
        # check that there are exactly numVersion versions
        # in the version history for an object
        vInfo = self.repo.getVersionInfo(obj)
        histId = vInfo.history_id
        vh = self.repo.getVersionHistory(histId)
        vids = vh.getVersionIds()
        #labels = vh.getLabels()
        self.assertEquals(len(vids), numVersions)

    def testPartialPublishingIsEconomic(self):
        # we've noticed that partial publishing is having some not so
        # nice side effects:
        # * every non-published object is being saved in the
        # VersionRepository again
        #
        # * every published object is being saved in the repository twice
        #
        # basically, all objects are being saved one time more than they
        # should
        self.setupStagingAreas()
        mkt = self.home.mkt
        mkt.setStagePath('public_website')
        mkt.invokeFactory('Folder', id='f1')
        mkt.invokeFactory('Folder', id='f2')
        # full-publish the stage a first time
        mkt.publishStage('0')
        # sanity checks:
        # there should be only version of each document in the
        # repository
        self._checkVersions(mkt.f1, 1)
        self._checkVersions(mkt.f2, 1)
        self.subcommit()
        # let's sleep a little...
        import time
        time.sleep(1)
        # ...change some stuff
        mkt.f2.edit(title="Folder number 2")
        self.assertEquals(list(mkt.getLastChangedPaths()),
                          ["/".join(mkt.f2.getPhysicalPath())])
        # ...and do a partial publishing
        mkt.publishLastChanged('1')
        # now there should be 1 version and 2 labels of f1...
        self._checkVersions(mkt.f1, 1, 2)
        # and 2 versions and 2 labels of f2...
        self._checkVersions(mkt.f2, 2)

    def test_worksWithCheckedOutObjects(self):
        # old versions of ESS left objects in the checked-out state
        # partialPublishing must work with them nevertheless
        self.setupStagingAreas()
        mkt = self.home.mkt
        mkt.setStagePath('public_website')
        mkt.invokeFactory('Folder', id='f1')
        mkt.invokeFactory('Folder', id='f2')
        # full-publish the stage a first time
        mkt.publishStage('0')
        self.subcommit()
        # let's sleep a little...
        import time
        time.sleep(1)
        # ...leave f1 checked out like old ESS would
        self.repo.checkoutResource(mkt.f1)
        mkt.f2.edit(title="Folder number 2")
        self.assertEquals(list(mkt.getLastChangedPaths()),
                          ["/".join(mkt.f2.getPhysicalPath())])
        # ...and do a partial publishing
        mkt.publishLastChanged('1')
        # check the final state is checked-in
        self.failIf(self.vt.isCheckedOut(mkt.f1),
                    "f1 is checked out, shoudl be checked in")

    def testRenamedObjectsGetNewVersions(self):
        # renamed object should get new version during partial publishing
        self.setupStagingAreas()
        mkt = self.home.mkt
        mkt.setStagePath('public_website')
        mkt.invokeFactory('Folder', id='f1')
        mkt.invokeFactory('Folder', id='d1')
        mkt.f1.invokeFactory('Document', 'f1d1')
        mkt.publishStage('0')
        self.subcommit()
        import time
        time.sleep(1)
        # get version info before the rename
        version_info = self.repo.getVersionInfo(mkt.d1)
        d1_info = (version_info.history_id, version_info.version_id)
        version_info = self.repo.getVersionInfo(mkt.f1.f1d1)
        f1d1_info = (version_info.history_id, version_info.version_id)
        # now let's rename
        mkt.manage_renameObject('d1', 'd2')
        mkt.f1.manage_renameObject('f1d1', 'f1d2')
        time.sleep(1)
        # and partial-publish
        mkt.publishLastChanged('2')
        # finally, let's check the versions changed
        version_info = self.repo.getVersionInfo(mkt.d2)
        d2_info = (version_info.history_id, version_info.version_id)
        version_info = self.repo.getVersionInfo(mkt.f1.f1d2)
        f1d2_info = (version_info.history_id, version_info.version_id)
        self.assertNotEquals(d1_info, d2_info)
        self.assertNotEquals(f1d1_info, f1d2_info)

def test_suite():
    from unittest import TestSuite, makeSuite
    suite = TestSuite()
    for testclass in (
        TestPublishing,
        TestPartialDeployment,
        ):
        suite.addTest(makeSuite(testclass))
    return suite

if __name__ == '__main__':
    framework(descriptions=1, verbosity=1)
