# EnSimpleStaging  |  Copyright(C), 2004, Enfold Systems, LLC
# see LICENSE.txt for details

import os, sys
from sets import Set

if __name__ == '__main__':
    execfile(os.path.join(sys.path[0], 'framework.py'))

from Testing import ZopeTestCase
from Testing.ZopeTestCase.functional import Functional
from Products.PloneTestCase import PloneTestCase

for product in ('Archetypes',
                'PortalTransforms',
                'ZopeVersionControl',
                'CMFStaging',
                'EnSimpleStaging',
                'MimetypesRegistry',
#                'SecureMailHost',
                ):
    PloneTestCase.installProduct(product)
del product

from cStringIO import StringIO
from Products.CMFCore.utils import getToolByName
from Products.Archetypes.tests.utils import makeContent
from Products.EnSimpleStaging.tests.base import TestStaging
from Products.EnSimpleStaging.tools.workspaces import StagingError

PloneTestCase.setupPloneSite()

class TestChangeSets(TestStaging):

    def afterSetUp(self):
        TestStaging.afterSetUp(self)
        self.setupStagingAreas()

    def testSimpleTaggingIsReversible(self):
        mkt = self.home.mkt
        public = self.public
        public.invokeFactory('Folder', id='marketing')
        mkt.setStagePath('public_website/marketing')
        mkt.invokeFactory('Folder', id='afolder')
        mkt.saveStage(label='initial')
        index, annotLast = mkt.getLastReversibleAnnotation()

    def testChangeSetsFullDeployment(self):
        mkt = self.home.mkt
        public = self.public
        public.invokeFactory('Folder', id='marketing')
        mkt.setStagePath('public_website/marketing')

        getChanges = mkt.getLastMovementsAndModifications

        # 0. Must start by making at least a tag.
        mkt.saveStage(label='initial')

        # 1. Create a document, publish stage, with a label. Should
        # catch the add for this document.
        mkt.invokeFactory('Document', id='doc1')
        mkt.publishStage(label='1')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('doc1',)])

        # 2. Create a folder, edit existing document, publish stage,
        # with a label. Should catch the folder's add and the document
        # modification.
        mkt.invokeFactory('Folder', id='folder1')
        mkt.doc1.setTitle('Document 1')
        mkt.publishStage(label='2')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('folder1',)])
        self.assertEquals(cs['modifications'], [('doc1',)])

        # 3. Create document inside folder, remove doc1, publish stage
        # with a label. Should catch the new document's add, and the
        # old document remove plus a folder1 change.
        mkt.folder1.invokeFactory('Document', id='doc11')
        mkt.manage_delObjects(ids=['doc1'])
        mkt.publishStage(label='3')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('folder1', 'doc11')])
        self.assertEquals(cs['modifications'], [('folder1',)])
        self.assertEquals(cs['removes'], [('doc1',)])

        # 4. Remove folder, add a new folder with the same name,
        # publish stage with a label. Need to catch that the new
        # folder is different from the old one, thus yielding a remove
        # for the folder1 and the contained doc11, and a add for
        # folder1 and doc2.
        mkt.manage_delObjects(ids=['folder1'])
        mkt.invokeFactory('Folder', id='folder1')
        mkt.invokeFactory('Document', id='doc2')
        mkt.publishStage(label='4')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('doc2',), ('folder1',)])
        self.assertEquals(cs['removes'], [('folder1',), ('folder1', 'doc11')])

    def testChangeSetsPartialDeployment(self):
        mkt = self.home.mkt
        public = self.public
        public.invokeFactory('Folder', id='marketing')
        mkt.setStagePath('public_website/marketing')

        getChanges = mkt.getLastMovementsAndModifications

        # 0. Must start by making at least a tag.
        mkt.saveStage(label='initial')

        # 1. Create a document, publish stage, with a label. Should
        # catch the add for this document.
        mkt.invokeFactory('Document', id='doc1')
        mkt.publishStage(label='1')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('doc1',)])

        # 2. Create a folder, edit existing document, publish stage,
        # with a label. Should catch the folder's add and the document
        # modification.
        mkt.invokeFactory('Folder', id='folder1')
        mkt.doc1.setTitle('Document 1')
        # doc is a CMFDefault document. doc.setTitle() doesn't trigger a
        # reindex
        mkt.doc1.reindexObject()
        mkt.publishLastChanged(label='2')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('folder1',)])
        # Should work since we reindexed doc1 above.
        self.assertEquals(cs['modifications'], [('doc1',)])

        # 3. Create document inside folder, remove doc1, publish stage
        # with a label. Should catch the new document's add, and the
        # old document remove plus a folder1 change.
        mkt.folder1.invokeFactory('Document', id='doc11')
        mkt.manage_delObjects(ids=['doc1'])
        mkt.publishLastChanged(label='3')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('folder1', 'doc11')])
        # Next one should work if the patch to _setOb got applied.
        self.assertEquals(cs['modifications'], [('folder1',)])
        self.assertEquals(cs['removes'], [('doc1',)])

        # 4. Remove folder, add a new folder with the same name,
        # publish stage with a label. Need to catch that the new
        # folder is different from the old one, thus yielding a remove
        # for the folder1 and the contained doc11, and a add for
        # folder1 and doc2.
        mkt.manage_delObjects(ids=['folder1'])
        mkt.invokeFactory('Folder', id='folder1')
        mkt.invokeFactory('Document', id='doc2')
        mkt.publishLastChanged(label='4')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('doc2',), ('folder1',)])
        self.assertEquals(cs['removes'], [('folder1',), ('folder1', 'doc11')])

    def testLastMovements(self):
        mkt = self.home.mkt
        mkt.setStagePath('public_website')

        # Publish empty stage. Need a initial tag.
        mkt.saveStage(label='initial')
        mkt.publishStage()

        getChanges = mkt.getLastMovementsAndModifications

        # Everything should be empty.
        info = getChanges()
        self.failIf(info['adds'])
        self.failIf(info['modifications'])
        self.failIf(info['removes'])

        # Added f1, f1/d11 and d1
        mkt.invokeFactory('Folder', 'f1')
        mkt.f1.invokeFactory('Document', 'd11', title='Document One')
        mkt.invokeFactory('Document', 'd1')
        mkt.publishLastChanged()

        info = getChanges()
        self.assertEquals(info['adds'],
                          [('d1',), ('f1',), ('f1', 'd11')])
        self.assertEquals(info['modifications'], [])
        self.assertEquals(info['removes'], [])

        # Added f2, f2/d21 and removed d1
        mkt.invokeFactory('Folder', 'f2')
        mkt.f2.invokeFactory('Document', 'd21')
        mkt.manage_delObjects(ids=['d1'])
        mkt.publishLastChanged()

        info = getChanges()
        self.assertEquals(info['adds'],
                          [('f2',), ('f2', 'd21')])
        self.assertEquals(info['modifications'], [])
        self.assertEquals(info['removes'], [('d1',)])

        # Renamed 'd21' to 'd21a', equals to remove 'd21' and add 'd21a'
        self.commit()
        mkt.f2.manage_renameObjects(ids=['d21'], new_ids=['d21a'])

        mkt.publishLastChanged()
        info = getChanges()
        self.assertEquals(info['adds'],
                          [('f2', 'd21a')])
        self.assertEquals(info['modifications'], [('f2',)])
        self.assertEquals(info['removes'], [('f2', 'd21',)])

        # Renamed 'f2' to 'f2a', equals to remove f2, f2/d21a and add
        # f2a and f2/d21a
        self.commit()
        mkt.manage_renameObjects(ids=['f2'], new_ids=['f2a'])

        mkt.publishLastChanged()
        info = getChanges()
        self.assertEquals(info['adds'],
                          [('f2a',), ('f2a', 'd21a')])
        self.assertEquals(info['modifications'], [])
        self.assertEquals(info['removes'], [('f2',), ('f2', 'd21a',)])

        # Make a minor modification to make sure it survives a publish.
        self.assertEquals(mkt.f1.d11.Title(), 'Document One')
        mkt.f1.d11.setTitle('Document Alpha One')
        self.assertEquals(mkt.f1.d11.Title(), 'Document Alpha One')

        # Full publish, mark everything as modified
        mkt.publishStage()
        info = getChanges()
        self.assertEquals(info['adds'], [])
        self.assertEquals(info['modifications'],
                          [('f1',), ('f1', 'd11'),
                           ('f2a',), ('f2a', 'd21a')])
        self.assertEquals(info['removes'], [])
        self.assertEquals(mkt.f1.d11.Title(), 'Document Alpha One')

    def testWFStateChangeSetsFullDeployment(self):
        mkt = self.home.mkt
        public = self.public
        public.invokeFactory('Folder', id='marketing')
        mkt.setStagePath('public_website/marketing')
        mkt.setDeployStates(('published',))
        doInPlace = mkt.doInPlace()

        wf = getToolByName(mkt, 'portal_workflow')
        getChanges = mkt.getLastMovementsAndModifications
        def status(ob):
            return wf.getInfoFor(ob, 'review_state', default=None)

        def action(ob, action):
            return wf.doActionFor(ob, action)

        # 0. Must start by making at least a tag.
        mkt.saveStage(label='initial')

        # 1. Create a document, publish stage, with a label. Should
        # not catch the add for this document because it isn't in
        # 'published' state.
        mkt.invokeFactory('Document', id='doc1')
        self.assertEquals(status(mkt.doc1), 'visible')

        mkt.publishStage(label='1')
        cs = getChanges()
        self.assertEquals(cs['adds'], [])

        # 2. Now, publish the document and then publish the stage. It
        # should now catch the add for this document.
        action(mkt.doc1, 'publish')
        self.assertEquals(status(mkt.doc1), 'published')

        mkt.publishStage(label='2')
        cs = getChanges()
        self.assertEquals(cs['adds'], [('doc1',)])

        # 3. Create a folder, edit existing document, publish stage,
        # with a label. Should not catch the folder's add because it's
        # not published, but should catch the document modification.
        mkt.invokeFactory('Folder', id='folder1')
        self.assertEquals(status(mkt.folder1), 'visible')

        mkt.doc1.setTitle('Document 1')
        self.assertEquals(status(mkt.doc1), 'published')

        mkt.publishStage(label='3')

        cs = getChanges()
        self.assertEquals(cs['adds'], [])
        self.assertEquals(cs['modifications'], [('doc1',)])

        # 4. Create document inside folder, publish it publish stage
        # with a label. Should catch the new document's add and raise
        # an exception because the parent folder is not published
        mkt.folder1.invokeFactory('Document', id='doc11')
        action(mkt.folder1.doc11, 'publish')
        self.assertEquals(status(mkt.folder1.doc11), 'published')

        if not doInPlace:
            # If not an inplace deployment, a StagingError happens
            # because the containers are checked.
            self.assertRaises(StagingError, mkt.publishStage, label='4')
        else:
            # in-place deployment doesn't check containers, because
            # obviously there are none to check. it won't find the add
            # of the subdocument though, because it won't be able to
            # find info about the parent folder.
            mkt.publishStage(label='4')

            cs = getChanges()
            self.assertEquals(cs['adds'], [])

        # 5. Now, publish containing folder and switch doc1 to
        # 'visible' state by doing a 'retract'. Should catch the add
        # of folder1 and doc11 and removal of doc1.
        action(mkt.folder1, 'publish')
        self.assertEquals(status(mkt.folder1), 'published')

        action(mkt.doc1, 'retract')
        self.assertEquals(status(mkt.doc1), 'visible')

        mkt.publishStage(label='5')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('folder1',), ('folder1', 'doc11')])
        self.assertEquals(cs['modifications'], [])
        self.assertEquals(cs['removes'], [('doc1',)])

        # 6. Remove folder, add a new folder with the same name,
        # publish stage with a label. Need to catch that the new
        # folder is different from the old one, thus yielding a remove
        # for the folder1 and the contained doc11, and a add for
        # folder1 and doc2.
        mkt.manage_delObjects(ids=['folder1'])

        mkt.invokeFactory('Folder', id='folder1')
        action(mkt.folder1, 'publish')
        self.assertEquals(status(mkt.folder1), 'published')

        mkt.invokeFactory('Document', id='doc2')
        action(mkt.doc2, 'publish')
        self.assertEquals(status(mkt.doc2), 'published')

        mkt.publishStage(label='6')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('doc2',), ('folder1',)])
        self.assertEquals(cs['removes'], [('folder1',), ('folder1', 'doc11')])

        # 7. Add a new document into folder1, should catch the add of
        # this doc only.
        mkt.folder1.invokeFactory('Document', id='doc12')
        action(mkt.folder1.doc12, 'publish')
        self.assertEquals(status(mkt.folder1.doc12), 'published')

        mkt.publishStage(label='7')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('folder1', 'doc12')])

        # 8. Change the state of the containing folder to 'visible',
        # should raise an exception because it's not possible to have
        # the contained documents in public if the container isn't
        # deployed.
        action(mkt.folder1, 'retract')
        self.assertEquals(status(mkt.folder1), 'visible')

        if not doInPlace:
            # If not an inplace deployment, a StagingError happens
            # because the containers are checked.
            self.assertRaises(StagingError, mkt.publishStage, label='8')
        else:
            # in-place deployment doesn't check containers, because
            # obviously there are none to check. it should find a
            # removal for folder1 and doc12.
            mkt.publishStage(label='8')

            cs = getChanges()
            self.assertEquals(cs['adds'], [])
            self.assertEquals(cs['removes'],
                              [('folder1',), ('folder1', 'doc12')])
            self.failIf('folder1' in public.marketing.objectIds())

    def testWFStateChangeSetsPartialDeployment(self):
        mkt = self.home.mkt
        public = self.public
        public.invokeFactory('Folder', id='marketing')
        mkt.setStagePath('public_website/marketing')
        mkt.setDeployStates(('published',))
        doInPlace = mkt.doInPlace()

        wf = getToolByName(mkt, 'portal_workflow')
        getChanges = mkt.getLastMovementsAndModifications
        def status(ob):
            return wf.getInfoFor(ob, 'review_state', default=None)

        def action(ob, action):
            return wf.doActionFor(ob, action)

        # 0. Must start by making at least a tag.
        mkt.saveStage(label='initial')

        # 1. Create a document, publish stage, with a label. Should
        # not catch the add for this document because it isn't in
        # 'published' state.
        mkt.invokeFactory('Document', id='doc1')
        self.assertEquals(status(mkt.doc1), 'visible')

        mkt.publishLastChanged(label='1')
        cs = getChanges()
        self.assertEquals(cs['adds'], [])

        # 2. Now, publish the document and then publish the stage. It
        # should now catch the add for this document.
        action(mkt.doc1, 'publish')
        self.assertEquals(status(mkt.doc1), 'published')

        mkt.publishLastChanged(label='2')
        cs = getChanges()
        self.assertEquals(cs['adds'], [('doc1',)])

        # 3. Create a folder, edit existing document, publish stage,
        # with a label. Should not catch the folder's add because it's
        # not published, but should catch the document modification.
        mkt.invokeFactory('Folder', id='folder1')
        self.assertEquals(status(mkt.folder1), 'visible')

        mkt.doc1.setTitle('Document 1')
        # doc is a CMFDefault document. doc.setTitle() doesn't trigger a
        # reindex
        mkt.doc1.reindexObject()
        self.assertEquals(status(mkt.doc1), 'published')

        mkt.publishLastChanged(label='3')

        cs = getChanges()
        self.assertEquals(cs['adds'], [])
        # Should work since we reindexed doc1 above.
        self.assertEquals(cs['modifications'], [('doc1',)])

        # 4. Create document inside folder, publish it publish stage
        # with a label. Should catch the new document's add and raise
        # an exception because the parent folder is not published
        mkt.folder1.invokeFactory('Document', id='doc11')
        action(mkt.folder1.doc11, 'publish')
        self.assertEquals(status(mkt.folder1.doc11), 'published')

        if not doInPlace:
            # If not an inplace deployment, a StagingError happens
            # because the containers are checked.
            self.assertRaises(StagingError, mkt.publishLastChanged, label='4')
        else:
            # in-place deployment doesn't check containers, because
            # obviously there are none to check. it won't find the add
            # of the subdocument though, because it won't be able to
            # find info about the parent folder.
            mkt.publishLastChanged(label='4')

            cs = getChanges()
            self.assertEquals(cs['adds'], [])

        # 5. Now, publish containing folder and switch doc1 to
        # 'visible' state by doing a 'retract'. Should catch the add
        # of folder1 and doc11 and removal of doc1.
        action(mkt.folder1, 'publish')
        self.assertEquals(status(mkt.folder1), 'published')

        action(mkt.doc1, 'retract')
        self.assertEquals(status(mkt.doc1), 'visible')

        mkt.publishLastChanged(label='5')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('folder1',), ('folder1', 'doc11')])
        self.assertEquals(cs['modifications'], [])
        self.assertEquals(cs['removes'], [('doc1',)])

        # 6. Remove folder, add a new folder with the same name,
        # publish stage with a label. Need to catch that the new
        # folder is different from the old one, thus yielding a remove
        # for the folder1 and the contained doc11, and a add for
        # folder1 and doc2.
        mkt.manage_delObjects(ids=['folder1'])

        mkt.invokeFactory('Folder', id='folder1')
        action(mkt.folder1, 'publish')
        self.assertEquals(status(mkt.folder1), 'published')

        mkt.invokeFactory('Document', id='doc2')
        action(mkt.doc2, 'publish')
        self.assertEquals(status(mkt.doc2), 'published')

        mkt.publishLastChanged(label='6')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('doc2',), ('folder1',)])
        self.assertEquals(cs['removes'], [('folder1',), ('folder1', 'doc11')])

        # 7. Add a new document into folder1, should catch the add of
        # this doc only.
        mkt.folder1.invokeFactory('Document', id='doc12')
        action(mkt.folder1.doc12, 'publish')
        self.assertEquals(status(mkt.folder1.doc12), 'published')

        mkt.publishLastChanged(label='7')

        cs = getChanges()
        self.assertEquals(cs['adds'], [('folder1', 'doc12')])

        # 8. Change the state of the containing folder to 'visible',
        # should raise an exception because it's not possible to have
        # the contained documents in public if the container isn't
        # deployed.
        action(mkt.folder1, 'retract')
        self.assertEquals(status(mkt.folder1), 'visible')

        # should find a removal for folder1 and doc12. what happens
        # here is that folder1 is deleted immediately from the target.
        mkt.publishLastChanged(label='8')

        cs = getChanges()
        self.assertEquals(cs['adds'], [])
        self.assertEquals(cs['removes'],
                          [('folder1',), ('folder1', 'doc12')])
        self.failIf('folder1' in public.marketing.objectIds())

    def testWFStateLastMovements(self):
        mkt = self.home.mkt
        mkt.setStagePath('public_website')
        mkt.setDeployStates(('published',))
        doInPlace = mkt.doInPlace()

        wf = getToolByName(mkt, 'portal_workflow')
        getChanges = mkt.getLastMovementsAndModifications
        def status(ob):
            return wf.getInfoFor(ob, 'review_state', default=None)

        def action(ob, action):
            return wf.doActionFor(ob, action)

        # Publish empty stage. Need a initial tag.
        mkt.saveStage(label='initial')
        mkt.publishStage()

        getChanges = mkt.getLastMovementsAndModifications

        # Everything should be empty.
        info = getChanges()
        self.failIf(info['adds'])
        self.failIf(info['modifications'])
        self.failIf(info['removes'])

        # Added f1, f1/d11 and d1
        mkt.invokeFactory('Folder', 'f1')
        mkt.f1.invokeFactory('Document', 'd11', title='Document One')
        mkt.invokeFactory('Document', 'd1')
        mkt.publishLastChanged()

        # Should see no changes, none of the objects was in
        # 'published' state.
        info = getChanges()
        self.assertEquals(info['adds'], [])
        self.assertEquals(info['modifications'], [])
        self.assertEquals(info['removes'], [])

        # Publish the objects
        action(mkt.f1, 'publish')
        action(mkt.f1.d11, 'publish')
        action(mkt.d1, 'publish')
        mkt.publishLastChanged()

        # Should see the changes now, all of the objects were in
        # 'published' state.
        info = getChanges()
        self.assertEquals(info['adds'],
                          [('d1',), ('f1',), ('f1', 'd11')])
        self.assertEquals(info['modifications'], [])
        self.assertEquals(info['removes'], [])

        # Added f2, f2/d21 and removed d1
        mkt.invokeFactory('Folder', 'f2')
        mkt.f2.invokeFactory('Document', 'd21')
        mkt.manage_delObjects(ids=['d1'])
        mkt.publishLastChanged()

        # Should see no changes except for d1, none of the other
        # objects was in 'published' state.
        info = getChanges()
        self.assertEquals(info['adds'], [])
        self.assertEquals(info['modifications'], [])
        self.assertEquals(info['removes'], [('d1',)])

        # Publish the objects
        action(mkt.f2, 'publish')
        action(mkt.f2.d21, 'publish')
        mkt.publishLastChanged()

        # Should see changes to f2 and f2/d1, which were now
        # published. d1 has already been removed by the last
        # deployment.
        info = getChanges()
        self.assertEquals(info['adds'],
                          [('f2',), ('f2', 'd21')])
        self.assertEquals(info['modifications'], [])
        self.assertEquals(info['removes'], [])

        # Renamed 'd21' to 'd21a', equals to remove 'd21' and add 'd21a'
        self.commit()
        mkt.f2.manage_renameObjects(ids=['d21'], new_ids=['d21a'])

        mkt.publishLastChanged()
        info = getChanges()
        self.assertEquals(info['adds'],
                          [('f2', 'd21a')])
        self.assertEquals(info['modifications'], [('f2',)])
        self.assertEquals(info['removes'], [('f2', 'd21',)])

        # Renamed 'f2' to 'f2a', equals to remove f2, f2/d21a and add
        # f2a and f2/d21a
        self.commit()
        mkt.manage_renameObjects(ids=['f2'], new_ids=['f2a'])

        mkt.publishLastChanged()
        info = getChanges()
        self.assertEquals(info['adds'],
                          [('f2a',), ('f2a', 'd21a')])
        self.assertEquals(info['modifications'], [])
        self.assertEquals(info['removes'], [('f2',), ('f2', 'd21a',)])

        # Make a minor modification to make sure it survives a publish.
        self.assertEquals(mkt.f1.d11.Title(), 'Document One')
        mkt.f1.d11.setTitle('Document Alpha One')
        self.assertEquals(mkt.f1.d11.Title(), 'Document Alpha One')

        # Full publish, mark everything as modified
        mkt.publishStage()
        info = getChanges()
        self.assertEquals(info['adds'], [])
        self.assertEquals(info['modifications'],
                          [('f1',), ('f1', 'd11'),
                           ('f2a',), ('f2a', 'd21a')])
        self.assertEquals(info['removes'], [])
        self.assertEquals(mkt.f1.d11.Title(), 'Document Alpha One')

        # Retract f2a/d21a, should show up as removed in next deployment.
        action(mkt.f2a.d21a, 'retract')

        # Full publish, mark everything as modified
        mkt.publishStage()
        info = getChanges()
        self.assertEquals(info['adds'], [])
        self.assertEquals(info['modifications'],
                          [('f1',), ('f1', 'd11'), ('f2a',)])
        self.assertEquals(info['removes'], [('f2a', 'd21a')])
        self.assertEquals(mkt.f1.d11.Title(), 'Document Alpha One')

class TestInPlaceChangeSets(TestChangeSets):

    def afterSetUp(self):
        TestChangeSets.afterSetUp(self)
        self.home.mkt.setInPlace(True)

def test_suite():
    from unittest import TestSuite, makeSuite
    suite = TestSuite()
    for testclass in (
        TestChangeSets,
        TestInPlaceChangeSets,
        ):
        suite.addTest(makeSuite(testclass))
    return suite

if __name__ == '__main__':
    framework(descriptions=1, verbosity=1)
