# EnSimpleStaging  |  Copyright(C), 2004, Enfold Systems, LLC
# see LICENSE.txt for details

from __future__ import nested_scopes

from difflib import SequenceMatcher
from sets import Set
from zope.interface import providedBy, directlyProvides, directlyProvidedBy

from DateTime import DateTime
from BTrees.IOBTree import IOBTree
from BTrees.Length import Length

from AccessControl import ClassSecurityInfo
from AccessControl.SecurityManagement import getSecurityManager
from Acquisition import aq_base
from Products.Archetypes.public import *
from Products.Archetypes.utils import findDict
from Products.ATContentTypes.content.folder import ATFolder
from Products.CMFCore.utils import getToolByName
from Products.CMFCore.CMFCorePermissions import View, ReviewPortalContent
from Products.CMFStaging.StagingTool import StageObjects
from Products.EnSimpleStaging.config import PROJECTNAME
from Products.EnSimpleStaging.interfaces import IInPlaceStaging

# JobServer hook:
try:
    from Products.JobServer.interfaces import IJobQueue
except ImportError:
    IJobQueue = None

class StagingArea(OrderedBaseFolder):
    """ A simple workspace implementation """

    security = ClassSecurityInfo()
    portal_type = 'StagingArea'
    meta_type = archetype_name = 'Staging Area'
    _at_rename_after_creation = True # rename object according to the title?
    _at_type_subfolder = 'Folder'

    allow_discussion = False
    filter_content_types  = 0
    allowed_content_types = ("Sandbox", )
    content_icon = "staging-area-icon.gif"

    actions = (#{'id': 'view',
               # 'name': 'View',
               # 'action': 'string:${object_url}/stagingarea_view',
               # 'permissions': (View,)},
               {'id': 'stage_map',
                'name': 'Stage Map',
                'action': 'string:${object_url}/changeset_tree',
                'permissions': (ReviewPortalContent,)},
               {'id': 'deployment',
                'name': 'Deployment',
                'action': 'string:${object_url}/workspace_deployment',
                'permissions': (ReviewPortalContent,)},
               )

    _stage_info = None
    _si_length = None

    schema = BaseFolder.schema + Schema((
        StringField(
        'stagePath',
	write_permission=ReviewPortalContent,
        widget=StringWidget(label='Deployment Path',
                            condition='not:object/doInPlace',
                            visible = {"edit":"visible", "view":"invisible" },
                            description=(
        'Path relative to the portal object, '
        'where content will be deployed into. '
        'I.e. public_website/sales '
        'NOTICE: the destination must exist. '
        'An empty value means this workspace is disabled and '
        'will not deploy at all.'),
                            ),
        ),
        BooleanField(
        'queueJobsByDefault',
	write_permission=ReviewPortalContent,
        default=False,
        widget=BooleanWidget(label="Queue deployments by default?",
                             description=('This switch only affects the default value of the '
                                          '"Queued?" checkbox on the Deployment tab. '
                                          '"Queueing" a deployment means that the deployment '
                                          'is not done immediately, but is queued into the JobServer job queue. '
                                          'Of course, this only happens if the job queue is installed.'),
                             visible = {"edit":"visible", "view":"invisible" },
                             ),
        ),
        LinesField(
        'deployStates',
	write_permission=ReviewPortalContent,
        accessor='getDeployStates',
        mutator='setDeployStates',
        default=(),
        widget=LinesWidget(label='Deploy States',
                           visible = {"edit":"visible", "view":"invisible" },
                           description=(
        'Only deploy objects in the following workflow states:'
        ),
                           ),
        ),
        LinesField(
        'typesWhiteList',
	write_permission=ReviewPortalContent,
        default=(),
        vocabulary='_ess_vocabularyAllTypes',
        widget=InAndOutWidget(label='Portal Types whitelist',
                              visible = {"edit":"visible", "view":"invisible" },
                              description=(
        'if any portal_type is selected here, only the selected types can '
        'be added inside the workspace. If nothing is listed here, no '
        'filtering is done'),
                                    ),
        ),
        ))

    def _resetAnnotations(self):
        self._stage_info = IOBTree()
        self._si_length = Length()

    def initializeArchetype(self, **kwargs):
        BaseFolder.initializeArchetype(self, **kwargs)
        self._resetAnnotations()

    security.declarePublic('doInPlace')
    def doInPlace(self):
        """ returns whether this workspace is in in-place mode or not """
        # Backwards compatibility
        bw = getattr(aq_base(self), 'inPlace', None)
        if bw is not None:
            return bw
        return IInPlaceStaging.providedBy(self)

    def setInPlace(self, value):
        provided = directlyProvidedBy(self)
        if not value:
            if IInPlaceStaging.providedBy(self):
                # Do we provide it?
                if IInPlaceStaging in provided:
                    # Marker interface, asking for removal, remove it
                    directlyProvides(self, provided - IInPlaceStaging)
                else:
                    # Might be set on the class, not much we can do.
                    return
            else:
                # Doensn't provide it, noop.
                return
        else:
            # Setting to inPlace.
            if not IInPlaceStaging.providedBy(self):
                # Does not provide yet, set using directlyProvides
                directlyProvides(self, provided, IInPlaceStaging)
            else:
                # Already provides it, skip.
                return


    security.declareProtected(ReviewPortalContent, 'setStagePath')
    def setStagePath(self, value):
        """ Sets the destination stage
        """
        # change the destination stage information
        field = self.Schema()['stagePath']
        value = str(value) # in case of unicode/marshall
        StringField.set(field, self, value)
        if not value:
            # the stage has been disabled for deployment
            return
        wt = getToolByName(self, 'portal_workspaces')
        wt.checkDuplicateStages()

    def _getSourceStageName(self):
        return 'src_%s' % self.UID()

    security.declarePrivate('ess_filterWhitelistedTypes')
    def ess_filterWhitelistedTypes(self, portal_types):
        """ filters what object types can actually be added inside
        the portal """
        whitelist = self.getTypesWhiteList()
        # if whitelist is empty, do not filter
        if not whitelist:
            return portal_types

        from sets import Set
        whitelist = Set(whitelist)
        filtered_portal_types = [ptype for ptype in portal_types
                                 if ptype.getId() in whitelist]
        return filtered_portal_types


    security.declarePrivate('_ess_vocabularyAllTypes')
    def _ess_vocabularyAllTypes(self):
        portal_types = getToolByName(self, 'portal_types')
        typelist = [(fti.title_and_id(), fti.getId())
                     for fti in portal_types.listTypeInfo()]
        typelist.sort()
        return DisplayList([(id, title) for title, id in typelist])

    security.declarePrivate('getSourceStageInfo')
    def getSourceStageInfo(self):
        ut = getToolByName(self, 'portal_url')
        title = self.title_or_id() + " (source)"
        return (self._getSourceStageName(), # id
                title, # title (duh :-)
                ut.getRelativeContentURL(self), # path
                )

    security.declarePrivate('getDestinationStageInfo')
    def getDestinationStageInfo(self):
        field = self.Schema()['stagePath']
        path = StringField.get(field, self)
        if not path:
            # no dest. path initialized in this object
            return None
        portal = getToolByName(self, 'portal_url').getPortalObject()
        title = self.title_or_id() + " (destination)"
        return ('dst_%s' % self.UID(), # id
                title,
                path,
                )

    security.declarePrivate('manage_afterAdd')
    def manage_afterAdd(self, item, container):
        BaseFolder.manage_afterAdd(self, item, container)
        wt = getToolByName(self, 'portal_workspaces')
        wt.checkDuplicateStages()

    def _globalLabel(self, label):
        # global labels must be unique among all labels in a site
        prefix = self.UID() + '-'
        if label.startswith(prefix):
            # It's already uniquefied
            return label
        return prefix + label

    security.declareProtected(StageObjects, 'getNextLabel')
    def getNextLabel(self):
        return "%s" % self._si_length()

    def getLastChangedPaths(self, query=None):
        # return paths for objects from the catalog
        # that were changed since the last publication or tagging
        annotIndex, annot = self.getLastReversibleAnnotation()
        lastPubDate = annot['date']
        return self.ess_getLastChangedPathsSince(lastPubDate, query=query)


    def getLastMovements(self):
        index, annotLast = self.getLastReversibleAnnotation()
        index, annotPrevious = self.getLastReversibleAnnotation(before=index)
        wt = getToolByName(self, 'portal_workspaces')
        stage = wt.getStageOf(self)
        historyIdMapPrevious = wt.getPath2HistoryIdMap(
            stage, annotPrevious['args']['label'])
        historyIdMapLast = wt.getPath2HistoryIdMap(
            stage, annotLast['args']['label'])
        differ = SequenceMatcher(None, historyIdMapPrevious, historyIdMapLast)
        changes = differ.get_opcodes()
        removes, adds = interpretOpcodes(changes,
                                         historyIdMapPrevious,
                                         historyIdMapLast)
        removes = ["/".join(path) for path, histId in removes]
        adds = ["/".join(path) for path, histId in adds]
        return removes, adds

    def getLastMovementsAndModifications(self):
        index, annotLast = self.getLastReversibleAnnotation()
        index, annotPrevious = self.getLastReversibleAnnotation(before=index)
        return self.getMovementsAndModifications(
            annotPrevious['args']['label'],
            annotLast['args']['label'])

    def getMovementsAndModifications(self, labelPrevious, labelLast):
        # An object is considered:
        #
        # - Removed if a (path, history_id) tuple
        #   is present in the previous label but not on this one
        #
        # - Added if the inverse happens
        #
        # - Changed if the (path, history_id) remains
        #   between labels, but the version_id changes
        #
        wt = getToolByName(self, 'portal_workspaces')
        stage = wt.getStageOf(self)

        # Normalize labels.
        labelPrevious = self._globalLabel(labelPrevious)
        labelLast = self._globalLabel(labelLast)

        # Compute version info map for previous label
        vInfoMapPrevious = wt.getPath2VersionInfoMap(
            stage, labelPrevious)

        # Create a dictionary to ease our job computing modifications.
        historyIdMapPrevious = dict(
            [((path, history_id), version_id)
             for path, history_id, version_id in vInfoMapPrevious])
        himPreviousSet = Set(historyIdMapPrevious.keys())

        vInfoMapLast = wt.getPath2VersionInfoMap(
            stage, labelLast)
        historyIdMapLast = dict(
            [((path, history_id), version_id)
             for path, history_id, version_id in vInfoMapLast])
        himLastSet = Set(historyIdMapLast.keys())

        # Calculate the movements
        removes = himPreviousSet - himLastSet
        adds = himLastSet - himPreviousSet
        # Now calculate the modifications:
        # & -> intersection for sets
        himModSet = himLastSet & himPreviousSet

        # We've reduced the space to search for modifications by doing
        # an intersection with the . That should give us a small speed win. It
        # also makes sure there's no dupes in the 'modifications' set
        # below.
        versionIdMapPrevious = []
        versionIdMapLast = []
        for key in himModSet:
            path, history_id = key
            versionIdMapPrevious.append((path, historyIdMapPrevious[key]))
            versionIdMapLast.append((path, historyIdMapLast[key]))
        vimPreviousSet = Set(versionIdMapPrevious)
        vimLastSet = Set(versionIdMapLast)

        # Now, we calculate the real modifications by checking if the
        # version_id changed between the last and previous sets.
        modifications = vimLastSet - vimPreviousSet
        removes, adds, modifications = [_getPathAndSort(seq) for seq
                                        in removes, adds, modifications]
        return dict(removes=removes, adds=adds, modifications=modifications)

    security.declareProtected(StageObjects, 'publishLastChanged')
    def publishLastChanged(self, label=None, message=None, queued=False):
        if queued:
            # delegate to JobServer if present
            self.queueCall('publishLastChanged', label=label, message=message)
            return

        stage_path = "/".join(self.getPhysicalPath())
        get_transaction().note(
            "Incremental deployment of stage '%s'" % stage_path)
        # first, let's get the deployment information
        wt = getToolByName(self, 'portal_workspaces')

        # now let's locate the changed objects...
        changedPaths = self.getLastChangedPaths()
        # and check them in
        wt.checkInObjects([self.unrestrictedTraverse(path)
                           for path in changedPaths],
                          message)

        # now, let's snapshot the current state so that we can
        # revert to it later, and calculate the moved objects
        # (removes, copies, renames)
        if not label:
            label = self.getNextLabel()

        globalLabel = self._globalLabel(label)
        tag_only = self.doInPlace()

        args = {'label':globalLabel,
                'message':message,
                'to_stage':None,
                'to_stage_path':None}
        info = {'operation': tag_only and 'tag' or 'partialPublish+tag',
                'args':args,
                'count':len(changedPaths), # only an aproximation
                'label': label}

        # Always save stage.
        self.saveStage(label, message, info)
        if tag_only:
            # Great, that saves us space and time!
            return

        si = wt.getStagesInfo()
        to_stage = si.get(self.getStagePath())

        __traceback_info__=(self.getId(), 'destination stage: ' +
                            str(to_stage), changedPaths, si)

        if to_stage is None:
            raise ValueError, "Destination stage is not set. Cannot proceed."
        to_stages = [to_stage[0]]

        # It's a publishing operation, update args info.
        args['to_stage'] = to_stage
        args['to_stage_path'] = self.getStagePath()

        # now let's get the movements
        removes, adds = self.getLastMovements()

        # find the duplicates between the changed objects and the
        # moved/copied ones
        pubPath = "/".join(self.getPhysicalPath())
        adds = [pubPath + '/' + path for path in adds]
        addSet = Set(adds)
        addSet.union_update(changedPaths)

        # let's update the newly added and changed objects
        paths = [path.split('/') for path in addSet]
        paths.sort()
        objs = [(self.unrestrictedTraverse(path), path) for path in paths]

        # now, filter the added objects to contain only objects in the
        # required states, if any state has been specified.
        states = self.getDeployStates()
        state_remove = []
        if states:
            states = tuple(states) + (None,)
            wf = getToolByName(self, 'portal_workflow')
            for obj, path in objs:
                state = wf.getInfoFor(obj, 'review_state', default=None)
                if state not in states:
                    # mark for removal if not in required state.
                    state_remove.append(path)

        # get the final list of objects.
        objs = [obj for obj, path in objs if path not in state_remove]

        # and the final list of removes
        l_pubPath = len(pubPath.split('/'))
        for path in state_remove:
            rel_path = path[l_pubPath:]
            if rel_path not in removes:
                removes.append('/'.join(rel_path))

        # let's remove what's in the way...
        self._removeFromDestination(removes)

        # and finally publish the modified objects.
        wt.publishObjects(objs, to_stages)

    def _removeFromDestination(self, paths):
        # remove a bunch of relative paths from the destination stage
        root = getToolByName(self, 'portal_url').getPortalObject()
        dest_stage = root.unrestrictedTraverse(self.getStagePath())
        marker = object()
        for path in paths:
            obj = dest_stage.unrestrictedTraverse(path, marker)
            if obj is not marker:
                obj.aq_parent.manage_delObjects([obj.getId()])

    security.declareProtected(StageObjects, 'publishStage')
    def publishStage(self, label=None, message=None,
                     wipe=True, hardWipe=False,
                     queued=False):
        """Publish this stage to the targeted deployment stage.
        """
        if queued:
            # delegate to JobServer if present
            self.queueCall('publishStage', label=label, message=message,
                           wipe=wipe, hardWipe=hardWipe)
            return

        stage_path = "/".join(self.getPhysicalPath())
        get_transaction().note("full deployment of stage '%s'" % stage_path)
        wt = getToolByName(self, 'portal_workspaces')
        if not label:
            label = self.getNextLabel()

        globalLabel = self._globalLabel(label)

        tag_only = self.doInPlace()
        args = {'label':globalLabel,
                'message':message,
                'to_stage':None,
                'to_stage_path':None}
        info = {'operation': tag_only and 'tag' or 'publish+tag',
                'args':args,
                'count':self.count(),
                'label': label}

        if tag_only:
            # Great! Saves us space and time.
            self.saveStage(label, message, info, force_tag=True)
            return

        si = wt.getStagesInfo()
        to_stage = si.get(self.getStagePath())

        __traceback_info__ = (self.getId(), 'destination stage' +
                              str(to_stage), si)

        if to_stage is None:
            raise ValueError, "Destination stage is not set. Cannot proceed."

        # It's a publishing operation, update args info.
        args['to_stage'] = to_stage
        args['to_stage_path'] = self.getStagePath()

        to_stages = [to_stage[0]]
        source = self
        from_stage = wt.getStageOf(source)
        assert from_stage is not None
        wt.recursivePublish(source, from_stage, to_stages,
                            label=globalLabel, message=message,
                            wipe=wipe, hardWipe=hardWipe)
        self.annotate(info)

    security.declarePublic('isQueueingEnabled')
    def isQueueingEnabled(self):
        jqueue = self.unrestrictedTraverse('/job_queue', None)
        return (IJobQueue is not None and
                IJobQueue.providedBy(jqueue))

    security.declarePrivate('queueCall')
    def queueCall(self, callname, *args, **kw):
        assert self.isQueueingEnabled(), "JobServer not present. Can't queue call"
        jqueue = self.unrestrictedTraverse('/job_queue')
        
        methodpath = '/'.join(self.getPhysicalPath()) + '/' + callname
        user = getSecurityManager().getUser()
        username = user.getUserName()
        userpath = "/".join(user.aq_parent.getPhysicalPath())
        dispatch = jqueue.manage_addProduct['JobServer']
        jr = dispatch.manage_addJobRequest(methodpath=methodpath,
                                           username=username,
                                           userpath=userpath,
                                           args=args,
                                           kw=kw)
        return jr

    security.declareProtected(StageObjects, 'saveStage')
    def saveStage(self, label, message=None, info=None, force_tag=False):
        """Save and label this stage
        """
        wt = getToolByName(self, 'portal_workspaces')

        __traceback_info__ = (self.getId(),)

        globalLabel = self._globalLabel(label)
        source = self
        from_stage = wt.getStageOf(source)
        assert from_stage is not None
        wt.recursiveTag(source, from_stage, label=globalLabel,
                        message=message, force_tag=force_tag)
        if info is None:
            args = {'label': globalLabel,
                    'message': message}
            info = {'operation': 'tag',
                    'args': args,
                    'count': self.count(),
                    'label': label}
        self.annotate(info)

    security.declarePublic('isReversibleAnnotation')
    def isReversibleAnnotation(self, annotation):
        """check if the annotation represents a reversible tag"""
        # 'tag' must be present in the annotation operation
        # and the global label must be valid
        wt = getToolByName(self, 'portal_workspaces')

        return ('tag' in annotation['operation'] and
                wt.isValidLabel(self._getSourceStageName(),
                                annotation['args']['label']))

    security.declareProtected(StageObjects, 'updateStage')
    def updateStage(self, label):
        """Update stage to a given label.
        """
        # label here is a globalLabel
        if not label:
            raise ValueError, 'Cannot revert to empty label'
        if label not in [annotation['args']['label']
                         for annotation in self.getAnnotations()
                         if self.isReversibleAnnotation(annotation)
                         ]:
            raise ValueError, 'Cannot revert to label "%s"' % label
        wt = getToolByName(self, 'portal_workspaces')
        source = self
        wt.recursiveRevertToLabel(source, label)
        args = {'label':label}
        annotation = {'operation': 'update',
                      'count':self.count(),
                      'args':args}
        self.annotate(annotation)

    security.declarePublic('ess_isUnderWorkspace')
    ess_isUnderWorkspace = True

    security.declareProtected(StageObjects, 'getCurrentChangeSet')
    def getCurrentChangeSet(self):
        """ informational method that returns a tree of
        modified/added/deleted objects since the last deployment.  This
        method should not be taken as authoritative, as it uses only
        cataloged objects as sources """
        try:
            index, annotLast = self.getLastReversibleAnnotation()
        except ValueError:
            return None
        wt = getToolByName(self, 'portal_workspaces')
        cat = getToolByName(self, 'portal_catalog')
        stage = wt.getStageOf(self)
        labelLast = self._globalLabel(annotLast['args']['label'])

        # Compute paths for last label
        lastPaths = Set(['/'.join(path) for path, history_id, version_id in
                         wt.getPath2VersionInfoMap(stage, labelLast)])
        # Compute paths and records for content currently in the stage
        pubpath = '/'.join(self.getPhysicalPath())
        pubpathprefixlen = len(pubpath + '/')
        currPaths = dict([(rec.getPath()[pubpathprefixlen:], rec)
                          for rec in cat(path=pubpath)
                          if rec.getPath() != pubpath])
        # Compute paths for objects changed since last update
        # a subset of currPaths keys
        query = None
        deploy_states = self.getDeployStates()
        if deploy_states:
            query = dict(review_state=deploy_states)
        changedPaths = Set([path[pubpathprefixlen:] for path in
                                 self.getLastChangedPaths(query=query)])

        # assume objects in lastPaths were removed unless they are
        # present in currPaths
        status_map = dict( [ (tuple(path.split('/')),
                              dict(path=path,
                                   id=path.split('/')[-1],
                                   url=None,
                                   status='removed'))
                             for path in lastPaths ] )
        for path, record in currPaths.items():
            if path not in lastPaths:
                status = 'added'
            elif path in changedPaths:
                status = 'changed'
                # we only consider 'changed' those objects that are
                # not present on the old paths, otherwise all 'added'
                # look like 'changed' based on their mod. time
            else:
                status = 'unchanged'
            split_path = tuple(path.split('/'))
            status_map[split_path] = dict(path=path,
                                          id=split_path[-1],
                                          url=record.getURL(),
                                          status=status,
                                          title=record.Title,
                                          portal_type=record.portal_type,
                                          )
        status_map = status_map.items()
        status_map.sort()

        # colapse the list into a tree
        result = []
        parent_subobjects_map = { () : result }
        for split_path, info in status_map:
            parent_path = split_path[:-1]
            parent_subobjects_map[parent_path].append(info)
            parent_subobjects_map[split_path] = info['subobjects'] = []

        return result

    security.declarePrivate('annotate')
    def annotate(self, info):
        mt = getToolByName(self, 'portal_membership')
        more_info = {'user':mt.getAuthenticatedMember().getId(),
                     'date':DateTime()}
        info.update(more_info)
        self._si_length.change(1)
        next = self._si_length()
        self._stage_info[next] = info

    security.declarePublic('getLastAnnotation')
    def getLastAnnotation(self):
        """ Get the last annotation
        """
        return self._stage_info[self._si_length()]

    security.declarePublic('getLastReversibleAnnotation')
    def getLastReversibleAnnotation(self, before=None):
        """ Get the last annotation that is also a tag operation
        """
        if before is None:
            start = self._si_length()
        else:
            start = before - 1
        for i in xrange(start, 0, -1):
            annot = self._stage_info[i]
            if self.isReversibleAnnotation(annot):
                return i, annot
        raise ValueError, "No remaining reversible operations"

    security.declarePublic('mayPublishIncrementally')
    def mayPublishIncrementally(self):
        try:
            self.getLastReversibleAnnotation()
        except ValueError:
            return False
        else:
            return True

    security.declarePublic('getAnnotations')
    def getAnnotations(self, count=None):
        """ Get a sequence of annotations.

        If the count argument is provided, the 'count'
        last annotations are returned.
        """
        start = 1
        last = self._si_length()
        if count is not None:
            start = last - count
        return [self._stage_info[i] for i in range(start, last+1)]

    security.declarePrivate('count')
    def count(self):
        res = Length()
        def add(obj):
            res.change(1)
        recurseVisit(self, add)
        return res()

def recurseVisit(obj, func):
    for item in obj.objectValues():
        func(item)
        if item.aq_explicit.isPrincipiaFolderish:
            recurseVisit(item, func)
    return

def interpretOpcodes(opcodes, ilist, jlist):
    removes = []
    adds = []
    for tag, i1, i2, j1, j2 in opcodes:
        if tag == 'equal':
            # we're not interested in the areas that match
            continue
        if tag == 'delete' or tag == 'replace':
            removes.extend(ilist[i1:i2])
        if tag == 'insert' or tag == 'replace':
            adds.extend(jlist[j1:j2])
    return removes, adds

def _getPathAndSort(sequence):
    sequence = [path for path, dummy in sequence]
    sequence.sort()
    return sequence

registerType(StagingArea, PROJECTNAME)

# XXX This is not used anymore. But we will leave it around so pickles
# don't break.
class Sandbox(OrderedBaseFolder):
    """ Staging Sandbox """

    meta_type = portal_type = archetype_name = 'Sandbox'
    global_allow = 0

# Don't register it anymore.
# registerType(Sandbox, PROJECTNAME)
