# This work is based upon CMFExternalFile, which was developed by
# Alan Runyan with contributions by Andy McKay, Kiran Jonnalagadda
# and PloneExternalFile developed by Florian Geier.
#
# CMFManagedFile extends CMFExternalFile to include a concept that
# can be found in PloneExternalFile where uploaded documents may be
# routed to specific directories or file systems.  This is specified by the
# user at upload time through the selection of pre-defined "repositories"
# that the site administrator would define and establish.
#
# The merger of CMFExternalFile and PloneExternalFile has been further
# extended to include some aspects of managing the external files on the
# file systems.  These include the automatic deletion of the file system files
# for deleted documents of the class CMFManagedFile governed by customizable
# and user selectable deletion policies, and the automatic movement of files
# from one file system repository to another based upon an edit action
# that includes the selection of a repository other than the repository
# in which the file is currently stored.
#
"""
$Id: trashcan.py 387 2004-11-15 17:33:56Z sidnei $
"""

import os
import time
import datetime
import OFS.Moniker

from Acquisition import aq_parent, aq_base
from AccessControl import ClassSecurityInfo
from Globals import InitializeClass
from ZODB.POSException import ConflictError

from OFS.Folder import Folder
from OFS.ObjectManager import BeforeDeleteException
from OFS.PropertyManager import PropertyManager
from OFS.SimpleItem import SimpleItem

from Products.CMFCore.CMFCorePermissions import ManagePortal, View
from Products.CMFCore.utils import getToolByName
from Products.PageTemplates.PageTemplateFile import PageTemplateFile

from config import DELETE_MANUAL, DELETE_DEFERRED
from config import TEMP_DOCUMENT, REUSED_FILENAME, TOOL_NAME


def folder_deleteHook(self, container):
    """Invoked as a result of deleting a repository folder
    """
    ftool = getToolByName(self, TOOL_NAME)
    self.purgeTemp(ftool.getTempDays())
    self.purgeDeferred(ftool.getDays())
    objects = self.objectIds()
    if len(objects):
        raise BeforeDeleteException, 'Repository folder contents has not expired'

def addTrashCan(context, id):
    obj = TrashCan(id)
    context._setObject(id, obj)
    return context._getOb(id)

class TrashCan(Folder):
    """ TrashCan
    """

    meta_type = 'ManagedFile TrashCan'

    security = ClassSecurityInfo()

    manage_options = (
        (Folder.manage_options[0],) +
        ({'label':'Overview', 'action':'manage_overview'},) +
        Folder.manage_options[1:]
        )

    def __init__(self, id=None):
        if id is not None:
            self.id = str(id)

    def manage_beforeDelete(self, item, container):
        folder_deleteHook(self, container)

    security.declareProtected(ManagePortal, 'purgeTemp')
    def purgeTemp(self, days):
        """Delete temporary objects/files if they have exceeded their
        alotted timeframe
        """
        delta = datetime.timedelta(days=days)
        cutoff = datetime.date.fromtimestamp(time.time()) - delta
        for o in self.objectValues():
            if (isinstance(o, DeletedFile) and
                o.deletePolicy in (TEMP_DOCUMENT,)):
                deleted = datetime.date.fromtimestamp(o.orphaned)
                if deleted <= cutoff:
                    self._delObject(o.id)

    security.declareProtected(ManagePortal, 'purgeDeferred')
    def purgeDeferred(self, days):
        """Delete files that have exceeded their alotted timeframes
        """
        delta = datetime.timedelta(days=days)
        cutoff = datetime.date.fromtimestamp(time.time()) - delta
        for o in self.objectValues():
            if (isinstance(o, DeletedFile) and
                o.deletePolicy in (DELETE_DEFERRED, DELETE_MANUAL)):
                deleted = datetime.date.fromtimestamp(o.orphaned)
                if deleted <= cutoff:
                    self._delObject(o.id)

    security.declareProtected(ManagePortal, 'checkReusableName')
    def checkReusableName(self, filepath):
        """ Protect newly added files against deletion where the
        file name has been reused
        """
        for o in self.objectValues():
            if isinstance(o, DeletedFile) and filepath == o.filepath:
                o.deletePolicy = DELETE_MANUAL
                o.filename = REUSED_FILENAME + o.filename

InitializeClass(TrashCan)

def deleteHook(self, container):
    """Invoked when repository instance is deleted to remove file
    from the file system.  For temporary files, the object is also deleted
    from the object database.
    """

    # Do nothing on copy
    if getattr(self, '_v_is_copy', False):
        return

    ftool = getToolByName(self, TOOL_NAME)

    try:
        if (self.deletePolicy != DELETE_MANUAL and
            self.filepath and
            os.path.isfile(self.filepath)):
            ftool.cleanupFiles(self.filepath)
    except ConflictError:
        raise
    except:
        pass

    if self.deletePolicy == TEMP_DOCUMENT:
        app = self.getPhysicalRoot()
        try:
            m = OFS.Moniker.loadMoniker(self.ZODBobject)
            ZODBobject = m.bind(app)
            ZODBfolder = aq_parent(ZODBobject)
            ZODBfolder._delObject(ZODBobject.id)
        except ConflictError:
            raise
        except:
            pass

def addDeletedFile(context, id, title, filename, deletePolicy):
    obj = DeletedFile(id, title, filename, deletePolicy)
    context._setObject(id, obj)
    return context._getOb(id)

class DeletedFile(PropertyManager, SimpleItem):
    """ Used to manage obsolete files in the file system
    repositories so that these files may be cleaned up.
    """

    meta_type = 'ManagedFile Deleted'
    security = ClassSecurityInfo()

    manage_options = (
        ({'label':'Content', 'action':'manage_workspace' },
         {'label':'Overview', 'action':'manage_overview' },) +
        PropertyManager.manage_options +
        SimpleItem.manage_options
        )

    def __init__(self, id, title, filename, deletePolicy):
        self.id = id
        self.title = title
        self.deletePolicy = deletePolicy
        self.orphaned = time.time()
        self.filepath = None
        self.filename = filename
        self.filesize = 0

    def manage_beforeDelete(self, item, container):
        deleteHook(self, container)
        # Clear the flag if it was set
        self._v_is_copy = False

    def _notifyOfCopyTo(self, container, op=0):
        self._v_is_copy = True
        SimpleItem._notifyOfCopyTo(self, container, op=op)

    def manage_workspace(self, URL1, RESPONSE):
        """View Repository"""
        RESPONSE.redirect(URL1 + '/cmfmf_view_rep')

    def get_size(self):
        return self.filesize

    size = get_size

    security.declareProtected(ManagePortal, 'setFSSize')
    def setFSSize(self, filesize):
        """Set file size of obsolete file
        """
        self.filesize = filesize

    security.declareProtected(ManagePortal, 'setFSPath')
    def setFSPath(self, filepath):
        """Set filesystem path to obsolete file
        """
        self.filepath = filepath

    security.declareProtected(View, 'getFSPath')
    def getFSPath(self):
        """Filesystem Path
        """
        return self.filepath

    security.declareProtected(View, 'getFSName')
    def getFSName(self):
        """Filename of obsolete file
        """
        return self.filename

    security.declareProtected(View, 'getFSTitle')
    def getFSTitle(self):
        """Title of obsolete file
        """
        return self.title

    security.declareProtected(View, 'getFSPolicy')
    def getFSPolicy(self):
        """Deletion policy of obsolete file
        """
        return self.deletePolicy

    security.declareProtected(View, 'getFSExpires')
    def getFSExpires(self):
        """Get file expiration date
        """
        ftool=getToolByName(self, TOOL_NAME)
        if self.deletePolicy != TEMP_DOCUMENT:
            seconds = int(ftool.getDays())
        else:
            seconds = int(ftool.getTempDays())
        expires = self.orphaned + (seconds*24*60*60)
        return time.strftime('%A %d-%B-%Y %H:%M:%S', time.localtime(expires))

InitializeClass(DeletedFile)
