# This work is based upon CMFExternalFile, which was developed by
# Alan Runyan with contributions by Andy McKay, Kiran Jonnalagadda
# and PloneExternalFile developed by Florian Geier.
#
# CMFManagedFile extends CMFExternalFile to include a concept that
# can be found in PloneExternalFile where uploaded documents may be
# routed to specific directories or file systems.  This is specified by the
# user at upload time through the selection of pre-defined "repositories"
# that the site administrator would define and establish.
#
# The merger of CMFExternalFile and PloneExternalFile has been further
# extended to include some aspects of managing the external files on the
# file systems.  These include the automatic deletion of the file system files
# for deleted documents of the class CMFManagedFile governed by customizable
# and user selectable deletion policies, and the automatic movement of files
# from one file system repository to another based upon an edit action
# that includes the selection of a repository other than the repository
# in which the file is currently stored.
#
"""
$Id: tool.py 525 2004-11-30 05:03:26Z sidnei $
"""

import os

from Acquisition import Implicit
from AccessControl import ClassSecurityInfo
from ComputedAttribute import ComputedAttribute
from Globals import InitializeClass, package_home

from OFS.SimpleItem import SimpleItem
from OFS.Folder import Folder
from OFS.ObjectManager import BeforeDeleteException

from Products.CMFCore.utils import getToolByName
from Products.CMFCore.utils import UniqueObject
from Products.CMFCore.CMFCorePermissions import ManagePortal, View
from Products.CMFCore.ActionProviderBase import ActionProviderBase
from Products.CMFCore.ActionsTool import ActionInformation as AI
from Products.CMFCore.Expression import Expression
from Products.PageTemplates.PageTemplateFile import PageTemplateFile

from batch import manage_addBatch
import trashcan
from policies import filename_policies, delete_policies, filesystem_policies
from config import FS_PHYSICAL, ORIGINAL_POLICY
from config import TOOL_NAME, DEFAULT_PATH, DELETE_MANUAL

class CMFManagedFileTool(UniqueObject, Folder, ActionProviderBase):
    """ The CMF Managed File tool
    """

    id = TOOL_NAME
    title = 'CMF ManagedFile Tool'
    meta_type = 'CMF ManagedFile Tool'

    security = ClassSecurityInfo()

    _actions = (
        AI(id='cmfmf_import',
           title='MF Import',
           description='Import a filesystem folder',
           action=Expression(text='string:cmfmf_import'),
           permissions=(ManagePortal,),
           category='folder',
           condition='',
           visible=1),)


    manage_options = (
        (Folder.manage_options[0],) +
        ({'label': 'Overview', 'action' : 'manage_overview'},
         {'label' : 'Configure', 'action' : 'manage_configure'},
         {'label' : 'Purge', 'action' : 'manage_purge'}) +
        ActionProviderBase.manage_options +
        SimpleItem.manage_options
        )


    security.declareProtected(ManagePortal, 'manage_overview')
    manage_overview = PageTemplateFile('www/explainTool', globals(),
                                       __name__='manage_overview')

    security.declareProtected(ManagePortal, 'manage_configure')
    manage_configure = PageTemplateFile('www/configureTool', globals(),
                                        __name__='manage_configure')

    security.declareProtected(ManagePortal, 'manage_purge')
    manage_purge = PageTemplateFile('www/purgeTool', globals(),
                                    __name__='manage_purge')

    _properties = (
        tuple(Folder._properties) +
        ({'id':'path_policy', 'type':'selection', 'mode': 'w',
          'select_variable':'availablePathPolicies'},
         {'id':'filename_policy', 'type':'selection', 'mode': 'w',
          'select_variable':'availableFilenamePolicies'},
         {'id':'basedir', 'type':'string', 'mode': 'w'},
         {'id':'purgeDays', 'type':'int', 'mode': 'w'},
         {'id':'tempDays', 'type':'int', 'mode': 'w'},))


    path_policy = FS_PHYSICAL
    filename_policy = ORIGINAL_POLICY
    purgeDays = 30
    tempDays = 2

    def __init__(self):
        Folder.__init__(self, self.id)

    basedir = DEFAULT_PATH

    security.declarePrivate('getBaseDir')
    def getBaseDir(self):
        """ Return the base directory setting
        """
        return self.getProperty('basedir', DEFAULT_PATH)

    def manage_afterAdd(self, item, container):
        res = Folder.manage_afterAdd(self, item, container)
        ids = self.objectIds()
        if not 'orphanedFiles' in ids:
            self.manage_addFolder('orphanedFiles', 'orphanedFiles')
        if not 'repository_container' in ids:
            import repository
            repository.addContainer(self)
        return res

    security.declareProtected(View, 'getDays')
    def getDays(self):
        """ Return the configured days before purging
        """
        return self.getProperty('purgeDays', 30)

    security.declareProtected(View, 'getTempDays')
    def getTempDays(self):
        """ Return the configured days for deleting temporary files
        """
        return self.getProperty('tempDays', 2)

    security.declareProtected(View, 'getPathPolicy')
    def getPathPolicy(self):
        """ Return the configured filesystem policy
        """
        return self.getProperty('path_policy', FS_PHYSICAL)

    security.declareProtected(View, 'getFilenamePolicy')
    def getFilenamePolicy(self):
        """ Return the configured filename policy
        """
        return self.getProperty('filename_policy', ORIGINAL_POLICY)

    security.declarePrivate('_repositoryList')
    def _repositoryList(self):
        base = self.repository_container
        repos = {}
        for repo in base.objectValues():
            repos[repo.getName()] = (repo.getPath(),
                                     repo.getPathPolicy(),
                                     repo.getFilenamePolicy(),
                                     repo.getProperty('visible'))
        return repos

    _repositoryList = ComputedAttribute(_repositoryList, 1)

    security.declareProtected(View, 'getRepositories')
    def getRepositories(self):
        """ Return the directory of configured repositories
        """
        repos = {}
        for k,v in self._repositoryList.items():
            if v[3]:
                repos[k] = v[:-1]
        return repos

    security.declareProtected(ManagePortal, 'edit')
    def edit(self, path_policy=None, filename_policy=None,
             days=None, temp=None, REQUEST=None):
        """ Edit the configured options on the tool
        """
        if path_policy is not None:
            self._updateProperty('path_policy', path_policy)
        if filename_policy is not None:
            self._updateProperty('filename_policy', filename_policy)
        if days is not None:
            self._updateProperty('purgeDays', days)
        if temp is not None:
            self._updateProperty('tempDays', temp)

        if REQUEST is not None:
            return self.manage_configure(
                self, REQUEST, management_view="Configure",
                manage_tabs_message="Changes saved.")

    security.declareProtected(ManagePortal, 'manage_editRepositories')
    def manage_editRepositories(self, chosen=(), add_repository=0,
                                delete_repository=0,
                                repository_name='', repository_policy='',
                                filename_policy='', repository_path='',
                                visible=False, REQUEST=None):
        """ Edit the repository list
        """
        rep = self._repositoryList
        base = self.repository_container

        if delete_repository:
            delete = []
            for repname in chosen:
                if repname in rep.keys():
                    self.delRepository(repname)

        if add_repository:
            if not repository_name and len(chosen) == 1:
                repository_name = chosen[0]
            if repository_name:
                if repository_name in rep.keys() and not repository_path:
                    repository_path = rep[repository_name][0]
                self.addRepository(repository_name,
                                   path=repository_path,
                                   path_policy=repository_policy,
                                   filename_policy=filename_policy,
                                   visible=visible)

        if REQUEST is not None:
            return self.manage_configure(
                self, REQUEST, management_view="Configure")

    security.declareProtected(ManagePortal, 'getRepository')
    def getRepository(self, name, default=None):
        base = self.repository_container
        return base._getOb(name, default)

    security.declareProtected(ManagePortal, 'getRepositoryInfo')
    def getRepositoryInfo(self, name, default=None):
        repo = self.getRepository(name, default)
        if repo == default:
            return default
        return (repo.getPath(),
                repo.getPathPolicy(),
                repo.getFilenamePolicy())

    security.declareProtected(ManagePortal, 'addRepository')
    def addRepository(self, name, path='', filename_policy='', path_policy='',
            visible=True):
        ob = self.getRepository(name)
        if ob is not None:
            ob.edit(path=path,
                    filename_policy=filename_policy,
                    path_policy=path_policy)
        else:
            import repository
            ob = repository.addRepository(self.repository_container,
                                          name, path=path,
                                          filename_policy=filename_policy,
                                          path_policy=path_policy,
                                          visible=visible)
        return ob

    security.declareProtected(ManagePortal, 'delRepository')
    def delRepository(self, name):
        base = self.repository_container
        return base._delObject(name)

    security.declareProtected(ManagePortal, 'bulkImportFilesystemPath')
    def bulkImportFilesystemPath(self, container, target_dir='',
                                 recursive=None,
                                 illegalChars='', dirNames='',
                                 dirPrefixes='', dirSuffixes='',
                                 fileNames='', filePrefixes='',
                                 fileSuffixes='', target_repository='default',
                                 delete_policy=DELETE_MANUAL,
                                 folder_type='Folder',
                                 file_type='ManagedFile'):
        """ Given a container and a path, bulk-load the local
        filesystem path contents into a container
        """
        return manage_addBatch(container, target_dir, recursive,
                               illegalChars, dirNames, dirPrefixes,
                               dirSuffixes, fileNames, filePrefixes,
                               fileSuffixes, target_repository,
                               delete_policy, folder_type, file_type)

    security.declareProtected(ManagePortal, 'getFilesystemPathFor')
    def getFilesystemPathFor(self, obj, filenameOverride='', createpath=True):
        """ Gets the physical location for the content and ensures
        the path creation if necessary
        """
        repo = self._repositoryList[obj.repository]
        path, repo_policy, file_policy, visible = repo
        component = filesystem_policies.component
        if not repo_policy:
            repo_policy = self.getPathPolicy()
        policy = component(repo_policy, context=obj, wrapped=True)
        physicalpath = policy.getPathFor(obj, filenameOverride, createpath)
        return physicalpath

    security.declareProtected(ManagePortal, 'manage_listPolicies')
    def manage_listPolicies(self):
        """ Return the repository policies
        """
        listing = []
        for info in filesystem_policies.info():
            item = {'id':info['name'],
                    'explanation':info['title']}
            if item['id'] == self.getPathPolicy():
                item['selected'] = True
            listing.append(item)
        return listing

    security.declareProtected(ManagePortal, 'availablePathPolicies')
    def availablePathPolicies(self, obj=None):
        # May return a different set of policies based on the obj.
        return tuple(filesystem_policies.keys())

    security.declareProtected(View, 'manage_listRepositories')
    def manage_listRepositories(self):
        """ Return a keyed directory of repositories
        """
        listing = []
        for id, entry in self._repositoryList.items():
            path = entry[0]
            policy = entry[1]
            filePolicy = entry[2]
            visible = entry[3]
            if not policy:
                policy = self._policy + " (default)"
            item = {'id':id, 'path':path,
                    'policy':policy, 'namePolicy':filePolicy,
                    'visible':visible,}
            listing.append(item)
        return listing

    security.declareProtected(View, 'manage_listRepositoryKeys')
    def manage_listRepositoryKeys(self):
        """ Return just the configured repository keys """
        return self._repositoryList.keys()

    security.declareProtected(ManagePortal, 'manage_listDeletePolicies')
    def manage_listDeletePolicies(self):
        """ Return a list of the deletion policies
        """
        return self.availableDeletePolicies()

    security.declareProtected(ManagePortal, 'availableDeletePolicies')
    def availableDeletePolicies(self, obj=None):
        # May return a different set of policies based on the obj.
        return tuple(delete_policies.keys())

    security.declareProtected(ManagePortal, 'manage_listFilenamePolicies')
    def manage_listFilenamePolicies(self):
        """ Return a list of the filename policies """
        return self.availableFilenamePolicies()

    security.declareProtected(ManagePortal, 'availableFilenamePolicies')
    def availableFilenamePolicies(self, obj=None):
        # May return a different set of policies based on the obj.
        return tuple(filename_policies.keys())

    # CMF Repository Action Tools
    security.declareProtected(ManagePortal, 'manage_addRepositoryFolder')
    def manage_addRepositoryFolder(self, obj, id):
        """ Add new repository folder to the repository
        """
        newFolder = trashcan.TrashCan(id)
        obj._setObject(id, newFolder)
        return obj._getOb(id)

    security.declareProtected(ManagePortal, 'manage_purgeRepository')
    def manage_purgeRepository(self):
        """ Purge the repository based upon configured options
        """
        baseFolder = self.orphanedFiles
        purgedFolders = 0
        remainingFolders = 0
        for o in baseFolder.objectValues():
            try:
                baseFolder._delObject(o.id)
                purgedFolders = purgedFolders + 1
            except BeforeDeleteException:
                remainingFolders = remainingFolders + 1
            except:
                raise
        message = ('%s folders purged, %s folders remain' %
                   (str(purgedFolders), str(remainingFolders)))
        return self.manage_purge(manage_tabs_message=message)

    security.declarePrivate('checkForReusedPath')
    def checkForReusedPath(self, filepath):
        """ Check deleted objects for reused repository file names
        """
        baseFolder = self.orphanedFiles
        for o in baseFolder.objectValues():
            try:
                o.checkReusableName(filepath)
            except:
                pass

    security.declarePrivate('cleanupFiles')
    def cleanupFiles(self, filepath):
        """ Remove file from file system and clean up emptied
        directories
        """
        try:
            os.remove(filepath)
        except:
            pass
        try:
            directory, child = os.path.split(filepath)
            while child:
                if not len(os.listdir(directory)):
                    os.rmdir(directory)
                else:
                    # No reason to keep looking up in the hierarchy
                    break
                directory, child=os.path.split(directory)
        except:
            pass

InitializeClass(CMFManagedFileTool)
