# repository.py | Copyright(C), 2004, Enfold Systems, LLC
#
# Enfold Systems, LLC
# 4500 Montrose Blvd., Suite D
# Houston, Texas 77006 USA
# p. +1 713.942.2377 | f. +1 832.201.8856
# www.enfoldsystems.com
# info@enfoldsystems.com
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License as
# published by the Free Software Foundation; either version 2.1 of the
# License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
# License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this library; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# For a full copy of the license, please see the included 'LGPL.txt' file.
"""
$Id: repository.py 536 2004-12-01 02:58:15Z arunyan $
"""

import os
from AccessControl import ClassSecurityInfo
from ComputedAttribute import ComputedAttribute
from Globals import InitializeClass
from OFS.SimpleItem import SimpleItem
from OFS.Folder import Folder
from OFS.PropertyManager import PropertyManager
from Products.CMFCore.utils import UniqueObject, getToolByName
from Products.CMFCore.CMFCorePermissions import View, ManagePortal
from Products.CMFManagedFile.interfaces import IRepository, IRepositoryContainer
from Products.CMFManagedFile.config import TOOL_NAME, DEFAULT_PATH, DIR_MODE
from Products.PageTemplates.PageTemplateFile import PageTemplateFile

def addContainer(context):
    obj = RepositoryContainer()
    id = obj.getId()
    context._setObject(id, obj)
    return context._getOb(id)

class RepositoryContainer(UniqueObject, Folder):
    """ A Container for Repositories
    """

    __implements__ = (Folder.__implements__ +
                      (IRepositoryContainer,))

    id = 'repository_container'
    meta_type = 'Repository Container'

    def all_meta_types(self):
        return Folder.all_meta_types(
            self, interfaces=(IRepository,))

    def manage_afterAdd(self, item, container):
        res = Folder.manage_afterAdd(self, item, container)
        if not 'default' in self.objectIds():
            addRepository(self, 'default')
        return res

InitializeClass(RepositoryContainer)

def addRepository(context, id, path='', filename_policy='',
                  path_policy='', visible=True, createpath=True):
    obj = Repository(id, path, filename_policy, path_policy, visible)
    context._setObject(id, obj)
    if createpath:
        if not path:
            path = DEFAULT_PATH
        if path.endswith(os.path.sep):
            path = path[:len(path)-len(os.path.sep)]
        if not os.path.exists(path):
            os.makedirs(path, mode=DIR_MODE)
    return context._getOb(id)


class Repository(SimpleItem, PropertyManager):
    """Repository Information
    """

    security = ClassSecurityInfo()

    __implements__ = (IRepository,)

    meta_type = 'File Repository'

    _properties = (
        tuple(PropertyManager._properties) +
        ({'id':'path', 'type':'string', 'mode':'w'},
         {'id':'filename_policy', 'type':'selection',
          'mode':'w', 'select_variable':'availableFilenamePolicies'},
         {'id':'path_policy', 'type':'selection',
          'mode':'w', 'select_variable':'availablePathPolicies'},
         {'id':'visible', 'type':'boolean',
          'mode':'w'},
         ))

    manage_options = (
        PropertyManager.manage_options +
        SimpleItem.manage_options
        )

    # Default value for backward compatibility
    visible = True

    def __init__(self, id, path='', filename_policy='', path_policy='',
            visible=True):
        self.id = id
        self.path = path
        self.filename_policy = filename_policy
        self.path_policy = path_policy
        self.visible = visible


    security.declareProtected(View, 'getName')
    def getName(self):
        return self.getId()

    security.declarePrivate('tool')
    def tool(self):
        tool = getToolByName(self, TOOL_NAME)
        return tool
    tool = ComputedAttribute(tool, 1)

    security.declareProtected(View, 'getPath')
    def getPath(self):
        path = self.getProperty('path') or self.tool.getBaseDir()
        if path.endswith(os.path.sep):
            path = path[:len(path) - len(os.path.sep)]
        return path

    security.declareProtected(View, 'getFilenamePolicy')
    def getFilenamePolicy(self):
        return (self.getProperty('filename_policy') or
                self.tool.getFilenamePolicy())

    security.declareProtected(View, 'getPathPolicy')
    def getPathPolicy(self):
        return (self.getProperty('path_policy') or
                self.tool.getPathPolicy())

    security.declareProtected(ManagePortal, 'edit')
    def edit(self, path=None, filename_policy=None,
             path_policy=None, visible=None, REQUEST=None):
        if path is not None:
            if path.endswith(os.path.sep):
                path = path[:len(path) - len(os.path.sep)]
            self._updateProperty('path', path)
        if filename_policy is not None:
            self._updateProperty('filename_policy', filename_policy)
        if path_policy is not None:
            self._updateProperty('path_policy', path_policy)
        if visible is not None:
            self._updateProperty('visible', visible)

        if REQUEST is not None:
            return self.manage_configure(
                self, REQUEST, management_view="Configure",
                manage_tabs_message="Changes saved.")

InitializeClass(Repository)

manage_addRepositoryForm = PageTemplateFile('www/addRepository', globals())
manage_addRepositoryForm.__name__ = 'addRepository'

def manage_addRepository(context, id, path='', filename_policy='',
                         path_policy='', visible=True, RESPONSE=None):
    """ add a repository """
    obj = addRepository(context, id, path, filename_policy, path_policy,
            visible)
    if RESPONSE is not None:
        RESPONSE.redirect(context.absolute_url() + '/manage_main')
