# This work is based upon CMFExternalFile, which was developed by
# Alan Runyan with contributions by Andy McKay, Kiran Jonnalagadda
# and PloneExternalFile developed by Florian Geier.
#
# CMFManagedFile extends CMFExternalFile to include a concept that
# can be found in PloneExternalFile where uploaded documents may be
# routed to specific directories or file systems.  This is specified by the
# user at upload time through the selection of pre-defined "repositories"
# that the site administrator would define and establish.
#
# The merger of CMFExternalFile and PloneExternalFile has been further
# extended to include some aspects of managing the external files on the
# file systems.  These include the automatic deletion of the file system files
# for deleted documents of the class CMFManagedFile governed by customizable
# and user selectable deletion policies, and the automatic movement of files
# from one file system repository to another based upon an edit action
# that includes the selection of a repository other than the repository
# in which the file is currently stored.
#
"""
$Id: registry.py 482 2004-11-24 17:29:13Z sidnei $
"""

import textwrap
from Acquisition import ImplicitAcquisitionWrapper
from Products.CMFManagedFile.log import info, debug
from Products.CMFManagedFile._exceptions import ComponentLookupError

def dotted_name(obj):
    klass = obj.__class__
    return '.'.join((klass.__module__, klass.__name__))

class RegistryItem:

    def __init__(self, component, name=None, title=None):
        if name is None:
            name = dotted_name(component)
        self.name = name
        if not title:
            title = getattr(component, '__doc__', '') or name
            title = textwrap.dedent(title.strip())
        self.title = title
        self.component = component

    def info(self):
        return {'name':self.name,
                'title':self.title}

    def __call__(self):
        return self.component

    def __repr__(self):
        comp_info = dotted_name(self.component)
        return '<%s instance registered as: %s>' % (comp_info, self.name)

class Registry(dict):

    factory = RegistryItem

    def register(self, component, name=None, title=None):
        item = self.factory(component=component, name=name, title=title)
        name = item.name
        if self.has_key(name):
            existing = self[name]
            info('Overriding %s with %s' % (existing, item))
        self[name] = item

    def info(self):
        for item in self.itervalues():
            yield item.info()

    def component(self, name, context=None, wrapped=False, default=None):
        if not self.has_key(name):
            return default
        component = self[name]()
        if not wrapped:
            return component
        if context is None:
            raise ComponentLookupError, ('The context argument is required '
                                         'for creating wrapped components')
        if hasattr(component, '__of__'):
            # Can be wrapped
            return component.__of__(context)
        return ImplicitAcquisitionWrapper(context, component)

    def __repr__(self):
        return '<%s with %s items>' % (self.__class__.__name__, len(self))
