# This work is based upon CMFExternalFile, which was developed by
# Alan Runyan with contributions by Andy McKay, Kiran Jonnalagadda
# and PloneExternalFile developed by Florian Geier.
#
# CMFManagedFile extends CMFExternalFile to include a concept that
# can be found in PloneExternalFile where uploaded documents may be
# routed to specific directories or file systems.  This is specified by the
# user at upload time through the selection of pre-defined "repositories"
# that the site administrator would define and establish.
#
# The merger of CMFExternalFile and PloneExternalFile has been further
# extended to include some aspects of managing the external files on the
# file systems.  These include the automatic deletion of the file system files
# for deleted documents of the class CMFManagedFile governed by customizable
# and user selectable deletion policies, and the automatic movement of files
# from one file system repository to another based upon an edit action
# that includes the selection of a repository other than the repository
# in which the file is currently stored.
#
"""
$Id: filesystem.py 482 2004-11-24 17:29:13Z sidnei $
"""

import os
from Acquisition import Implicit
from AccessControl import ClassSecurityInfo
from Globals import InitializeClass
from Products.CMFManagedFile import config
from Products.CMFCore.utils import getToolByName
from Products.CMFCore import CMFCorePermissions as perms
from Products.ExternalFile.FileUtils import safe_path_join
from _policies import filesystem_policies, filename_policies

register = filesystem_policies.register

class PathPolicy(Implicit):
    """ These are responsible for creating the paths
    """

    security = ClassSecurityInfo()

    security.declareProtected(perms.ModifyPortalContent, 'getPathFor')
    def getPathFor(self, obj, filenameOverride='', createpath=False):
        """ Creates a repository path string based upon a defined policy
        and creates the directories if necessary
        """

        r_name = getattr(obj, 'repository', '')
        relative_path = getattr(obj, 'basedir', '')
        filename = getattr(obj, 'filename', '') or obj.getId()

        prefix = self._getPath(obj)
        if prefix:
            prefix = safe_path_join(*prefix)
            relative_path = safe_path_join(prefix, relative_path)
        full_path, relative_path, filename = self.generatePathFor(r_name,
                                                                  relative_path,
                                                                  filename)
        folder = os.path.dirname(full_path)

        if createpath and not os.path.exists(folder):
            savecwd = os.getcwd()
            self.makeDir(folder)
            os.chdir(savecwd)

        if filenameOverride:
            filename = filenameOverride

        return safe_path_join(folder, filename)

    security.declareProtected(perms.ModifyPortalContent, 'generatePathFor')
    def generatePathFor(self, repository, relative_path, filename):
        """ Creates a repository path string based upon a defined policy
        """

        if not repository:
            repository = 'default'

        tool = getToolByName(self, config.TOOL_NAME)
        repo = tool.getRepository(repository, None)

        if repo is None:
            raise KeyError, repository

        base_path = repo.getPath()

        folder = safe_path_join(base_path, relative_path)

        fname_policy = repo.getFilenamePolicy()

        getComponent = filename_policies.component
        policy = getComponent(fname_policy)
        if policy is not None:
            filename = policy(filename)

        full_path = safe_path_join(folder, filename)
        return full_path, relative_path, filename


    security.declareProtected(perms.ModifyPortalContent, 'makeDir')
    def makeDir(self, path):
        """ Ensure all directories in path are created """
        os.makedirs(path, mode=config.DIR_MODE)


InitializeClass(PathPolicy)

class BasePathPolicy(PathPolicy):
    """
    Policy which stores objects in the repository base directory
    """

    security = ClassSecurityInfo()

    security.declarePrivate('_getPath')
    def _getPath(self, obj):
        return ()

InitializeClass(BasePathPolicy)
register(BasePathPolicy(), name=config.FS_BASE)

class RelativePathPolicy(PathPolicy):
    """
    Policy which stores objects in basedir/$relative_path/
    """

    security = ClassSecurityInfo()

    security.declarePrivate('_getPath')
    def _getPath(self, obj):
        url_tool=getToolByName(self, 'portal_url')
        return url_tool.getRelativeContentPath(obj)[:-1]

InitializeClass(RelativePathPolicy)
register(RelativePathPolicy(), name=config.FS_RELATIVE)

class PhysicalPathPolicy(PathPolicy):
    """
    Policy which stores objects in basedir/$physical_path/
    """

    security = ClassSecurityInfo()

    security.declarePrivate('_getPath')
    def _getPath(self, obj):
        return obj.getPhysicalPath()[:-1]

InitializeClass(PhysicalPathPolicy)
register(PhysicalPathPolicy(), name=config.FS_PHYSICAL)

class PortalTypePolicy(PathPolicy):
    """
    Policy which stores objects in basedir/$portal_type/
    """

    security = ClassSecurityInfo()

    security.declarePrivate('_getPath')
    def _getPath(self, obj):
        """ """
        return (obj.getTypeInfo().getId(), )

InitializeClass(PortalTypePolicy)
register(PortalTypePolicy(), name=config.FS_PORTAL_TYPE)
