# This work is based upon CMFExternalFile, which was developed by
# Alan Runyan with contributions by Andy McKay, Kiran Jonnalagadda
# and PloneExternalFile developed by Florian Geier.
#
# CMFManagedFile extends CMFExternalFile to include a concept that
# can be found in PloneExternalFile where uploaded documents may be
# routed to specific directories or file systems.  This is specified by the
# user at upload time through the selection of pre-defined "repositories"
# that the site administrator would define and establish.
#
# The merger of CMFExternalFile and PloneExternalFile has been further
# extended to include some aspects of managing the external files on the
# file systems.  These include the automatic deletion of the file system files
# for deleted documents of the class CMFManagedFile governed by customizable
# and user selectable deletion policies, and the automatic movement of files
# from one file system repository to another based upon an edit action
# that includes the selection of a repository other than the repository
# in which the file is currently stored.
#
"""
$Id: content.py 549 2004-12-02 00:55:53Z arunyan $
"""

import os
import string
import time
import OFS.Moniker
from exceptions import IOError
from StringIO import StringIO
from DateTime import DateTime
from random import random

from AccessControl import ClassSecurityInfo
from Globals import InitializeClass, package_home
from Acquisition import aq_parent, aq_base
from ZODB.POSException import ConflictError

from Products.CMFCore.CMFCorePermissions import View, ModifyPortalContent
from Products.CMFDefault.utils import formatRFC822Headers, html_headcheck
from Products.CMFDefault.utils import SimpleHTMLParser, bodyfinder
from Products.CMFDefault.utils import parseHeadersBody
from Products.CMFCore.utils import keywordsplitter, getToolByName

from Products.CMFCore.PortalContent import PortalContent
from Products.CMFDefault.DublinCore import DefaultDublinCoreImpl
from OFS.ObjectManager import ObjectManager
from OFS.Folder import Folder

from Products.ExternalFile.ExternalFile import ExternalFile
from Products.ExternalFile.FileUtils import clean_filename, copy_file
from Products.ExternalFile.FileUtils import extract_filename, safe_path_join
from Products.ExternalFile.ProductProperties import getMimeTypeFromFile

from config import DELETE_IMMEDIATE, DELETE_DEFERRED
from config import DELETE_MANUAL, TEMP_DOCUMENT, GENERATE_POLICY
from config import TOOL_NAME
from policies import filesystem_policies
import trashcan


InvalidCMFMFParameters = 'Invalid or missing parameters'

factory_type_information = {
    'id': 'Managed File',
    'content_icon': 'site_icon.gif',
    'meta_type': 'CMF ManagedFile',
    'description': ('Documents whose contents should be stored on an '
                    'external file system as opposed to within the ZODB.'),
    'product': 'CMFManagedFile',
    'factory': 'addCMFManagedFile',
    'immediate_view': 'cmfmf_edit_form',
    'actions': ({'id': 'view',
                 'name': 'View',
                 'action': 'string:${object_url}/file_view',
                 'permissions': (View,)
                 },
                {'id': 'download',
                 'name': 'Download',
                 'action': 'string:${object_url}/file_download',
                 'permissions'   : (View,)
                 },
                {'id': 'edit',
                 'name': 'Edit',
                 'action': 'string:${object_url}/cmfmf_edit_form',
                 'permissions': (ModifyPortalContent,)
                 },
                {'id': 'metadata',
                 'name': 'Metadata',
                 'action': 'string:${object_url}/metadata_edit_form',
                 'permissions'   : (ModifyPortalContent,)
                 }
                )}


def uniqueSuffix(useDate=1):
    """ Generate a unique string to be used as a unique suffix
    """
    now = DateTime()
    if useDate:
        time='%s.%s' % (now.strftime('%Y-%m-%d'), str(now.millis())[7:])
    else:
        time='%s' % str(now.millis())[7:]
    rand = str(random())[2:6]
    return time + rand


def addManagedFile(container,
                      id,
                      title='',
                      description='',
                      target_filepath='',
                      repository=None,
                      upload_file=None,
                      delete_policy=DELETE_MANUAL,
                      REQUEST=None,
                      factory=None):
    """ This method may be used to create CMF managed file object
    references to either "uploaded" content or to existing files on a
    file system available to the Plone server.  The behavior of this
    method relative to what is done with the external file content
    depends upon how the target_filepath, repository, and upload_file
    parameters are set.  You must identify the new content (file) by
    either specifying the target_filepath or the upload_file
    parameters.

    The target_filepath parameter is used to specify the relative path
    to a file inside a repository.  The repository is *required*. If a
    repository is not informed, we assume the default repository. The
    upload_file parameter is used to identify an uploaded file that
    should be stored on the file system.  **NOTE** that if both the
    target_filepath and the upload_file parameters are both specified,
    then the upload_file content will be stored in the file specified
    by the target_filepath parameter.  If this file already exists, it
    will be overwritten.

    Here are the basic behaviors:

    o Only the target_filepath parameter is specified - a
      CMFManagedFile instance is created, pointing to the file
      specified by default repository path + target_filepath.

    o target_filepath and repository parameters are specified - A
      CMFManagedFile instance is created pointing to the file
      specified by the repository path + target_filepath.

    o Only the upload_file parameter is specified - the filename
      attribute of the upload_file parameter object will be used to
      identify the name of the file, and the default repository will
      be used to determine the file system path.  The upload_file
      content will be written to the path specified by the repository
      under the file name derived from the filename attribute.

    o upload_file and repository parameters are specified - the
      filename attribute of the upload_file parameter object will be
      used to identify the name of the file, and the repository
      parameter will be used to determine the file system path.  The
      upload_file content will be written to the path specified by
      the repository under the file name derived from the filename
      attribute.

    o target_filepath and upload_file parameters are specified - the
      content of the upload_file object will be written out to the
      file specified by the target_filepath parameter.
    """
    if factory is None:
        factory = CMFManagedFile
    rel_path, filename, repository = startAdd(container, id, title,
                                              description, target_filepath,
                                              repository, upload_file,
                                              delete_policy, REQUEST)
    obj = factory(id, title, description)
    obj = finishAdd(container, obj, id, filename, rel_path,
              repository, upload_file, delete_policy)
    return id

def addCMFManagedFile(*args, **kw):
    addManagedFile(*args, **kw)

def startAdd(container, id, title, description, target_filepath, repository,
             upload_file, delete_policy, REQUEST):

    filename = ''
    rel_path = ''
    generatedName = ''
    ftool = getToolByName(container, TOOL_NAME)

    if repository is None:
        repository = 'default'

    # target_filepath is either a source (if upload not set) or a
    # target (if upload is set). It's always relative to the
    # repository (if one wasn't passed in, to the default repository)
    if target_filepath:
        norm_path = os.path.normpath(target_filepath)
        parts = os.path.split(norm_path)
        rel_path = os.path.join(*parts[:-1])
        filename = parts[-1]

    if upload_file:
        # if upload set, copy uploaded data to target file system file
        filename = extract_filename(upload_file)
        filename = clean_filename(filename)
        filename = os.path.basename(filename)

    return rel_path, filename, repository


def finishAdd(container, obj, id, filename, rel_path,
              repository, upload_file, delete_policy):
    # set additional metadata for this action
    obj._updateProperty('filename', filename)
    obj._updateProperty('basedir', rel_path)
    obj._updateProperty('repository', repository)
    ct = getMimeTypeFromFile(filename)
    obj._updateProperty('content_type', ct)

    # add to the folder
    container._setObject(id, obj)
    obj = container._getOb(id)

    # get the real computed path and make sure it exists
    ftool = getToolByName(container, TOOL_NAME)
    full_path = ftool.getFilesystemPathFor(obj)
    file_path = obj.getFilepath()
    # XXX not sure if this assert should be here since full_path ends up
    # replacing file_path
    #assert full_path == file_path, (full_path, file_path)

    obj._updateProperty('filename', os.path.basename(full_path))
    obj._updateProperty('filepath', full_path)
    if upload_file:
        copy_file(upload_file, full_path)

    # if this is a temporary document, artificially drive manage_beforeDelete
    if delete_policy == TEMP_DOCUMENT:
        repobj = deleteHook(obj, container)
        if repobj is not None:
            repobj.deletePolicy = delete_policy
            m = OFS.Moniker.Moniker(obj)
            repobj.ZODBobject = m.dump()

    obj.deletePolicy = delete_policy
    obj.reindexObject()
    return obj

def deleteHook(self, container):
    """ Store orphaned file in TOOL_NAME tool for cleanup
    """

    # Do nothing on copy
    if getattr(self, '_v_is_copy', False):
        return

    if self.deletePolicy == TEMP_DOCUMENT or not self.filepath:
        return

    # cannot say I understand why some instances of this fail
    ptool = getToolByName(self, 'portal_url', None)
    if ptool is None:
        ptool = getToolByName(container, 'portal_url')
    ftool = getToolByName(self, TOOL_NAME, None)
    if ftool is None:
        ftool = getToolByName(container, TOOL_NAME)

    # for DELETE_IMMEDIATE, cleanup the file system now and exit
    filepath = self.getFilepath()
    try:
        if (self.deletePolicy == DELETE_IMMEDIATE and
            filepath and
            os.path.isfile(filepath)):
            ftool.cleanupFiles(filepath)
            return
    except ConflictError:
        raise
    except:
        pass

    # create the tool repository folder and repository instance for this file
    orphanedRoot = ftool.orphanedFiles
    folder_id = time.strftime('%Y.%m', time.localtime(time.time()))
    orphaned = getattr(orphanedRoot, folder_id, None)
    if orphaned is None:
        orphaned = trashcan.addTrashCan(orphanedRoot, folder_id)
    rel_url = ptool.getRelativeContentURL(self)
    new_id = '.'.join((rel_url, self.filename, uniqueSuffix(useDate=0)))
    new_id = new_id.replace('\\', '.').replace('/', '.').replace(':', '.')
    new_file = trashcan.DeletedFile(new_id,
                                    self.title,
                                    self.filename,
                                    self.deletePolicy)
    new_file.setFSPath(self.filepath)
    new_file.setFSSize(self.size())
    orphaned._setObject(new_id, new_file)
    return new_file


class ManagedFileMixin(ExternalFile):
    """Mixin for Managed File implementations
    """

    __implements__ = (ExternalFile.__implements__,)

    # Declarative security
    security = ClassSecurityInfo()
    security.declareObjectProtected(View)

    _properties = tuple(ExternalFile._properties) + (
        {'id':'filename', 'type':'string', 'mode': 'w'},
        {'id':'basedir',  'type':'string', 'mode': '' },
        {'id':'repository', 'type':'string', 'mode': '' },
        {'id':'deletePolicy',   'type':'selection', 'mode': 'w',
         'select_variable':'getDeletePolicies'},
        )

    def __init__(self, id, title='', description='', filesystempath=''):
        ExternalFile.__init__(self, id, title, description, filesystempath)
        self.id = id
        self.repository = None
        self.filename = None
        self.deletePolicy = DELETE_MANUAL

    def manage_beforeDelete(self, item, container):
        deleteHook(self, container)
        # Clear the flag if it was set
        self._v_is_copy = False

    def _notifyOfCopyTo(self, container, op=0):
        if self.getProperty('deletePolicy') in (TEMP_DOCUMENT,):
            # Temporary documents can't be copied or renamed
            raise TypeError, 'Cannot copy or rename temporary documents'
        self._v_is_copy = True
        ExternalFile._notifyOfCopyTo(self, container, op=op)

    security.declareProtected(View, 'getRepositories')
    def getRepositories(self):
        """Return a list of all repository keys
        """
        ftool = getToolByName(self, TOOL_NAME)
        return ftool.getRepositories().keys()

    security.declareProtected(View, 'getDeletePolicies')
    def getDeletePolicies(self):
        """Return a list of all deletion policies
        """
        ftool = getToolByName(self, TOOL_NAME)
        return ftool.availableDeletePolicies(self)

    security.declarePrivate('manage_upload')
    def manage_upload(self, upload_file):
        self.edit(file=upload_file)

    def __str__(self, REQUEST=None):
        """
        If ExternalFile does not have contents or the file is missing
        it will return a IOError; this can not propagate any further
        """
        contents = ExternalFile.__call__(self, REQUEST=REQUEST)
        if isinstance(contents, IOError):
            contents = ''
        return contents

    security.declareProtected(View, 'getFilepath')
    def getFilepath(self, REQUEST=None):
        """
        Return the full pathname of the external file to which we are
        referring.  The REQUEST parameter encapsulates information
        about the environment which can be used by subclasses
        overriding this method.
        """
        tool = getToolByName(self, TOOL_NAME)
        fspath = tool.getFilesystemPathFor
        filename = self.getProperty('filename')
        return fspath(self, filenameOverride=filename, createpath=False)

    security.declarePublic('__len__')
    def __len__(self):
        """
        return length of object.  This is invoked by the Python
        builtin len(), which in turn is invoked by 'if foo:' where foo
        is an instance of this class.  Unless this method exists and
        returns a non-zero value, this locution will yield unexpected
        results.
        """
        fpath = self.getFilepath()
        if os.path.exists(fpath):
            return os.stat(fpath)[6]
        else:
            return 1 # non-zero to avoid unintended consequences


    def get_size(self):
        if os.path.isdir(self.getFilepath(self.REQUEST)):
            return 0
        return ExternalFile.get_size(self)

    size = get_size

    security.declareProtected(View, 'data')
    def data(self):
        """
        Compatibiliy with CMFDefault/File.py
        """
        try:
            if self.filepath:
                return self.getContents(self.REQUEST)
        except: pass
        return ''

    security.declareProtected(View, 'file_download')
    def file_download(self, client=None, REQUEST=None, **kw):
        """ """
        if self.get_size()==0:
            return ''

        # so that external file can set the headers
        if REQUEST is None:
            REQUEST = self.REQUEST

        RESPONSE = REQUEST.RESPONSE

        id = getattr(self, 'filename', self.getId())
        if not id:
            id = self.getId()
        RESPONSE.setHeader('Content-Disposition',
                           'attachment; filename=%s' % id)

        return ExternalFile.__call__(self, client=client, REQUEST=REQUEST, **kw)

    security.declareProtected(View, 'download')
    def download(self, client=None, REQUEST=None, **kw):
        """ """
        return self.file_download(client, REQUEST)

    security.declareProtected(View, 'index_html')
    index_html = file_download

    security.declareProtected(ModifyPortalContent, 'edit')
    def edit(self, precondition='', file='', repository='', delete_policy=''):
        """ Update and reindex.
        """

        oldRepository = self.getProperty('repository')
        oldFilepath = self.getFilepath()
        newRepository = repository
        ftool = getToolByName(self, TOOL_NAME)
        fspath = ftool.getFilesystemPathFor
        # if going into a new repository
        if (newRepository and
            newRepository != oldRepository):
            self._updateProperty('repository', newRepository)
            fp = fspath(self)
            self._updateProperty('filepath', fp)
            self._updateProperty('filename', os.path.basename(fp))

        if not file:
            file_path = self.getFilepath()
            # if only a repository move...
            if oldFilepath and oldFilepath != file_path:
                ftool.checkForReusedPath(file_path)
                copy_file(oldFilepath, file_path)
                ftool.cleanupFiles(oldFilepath)
        else:
            # self.filename could be empty if we are being uploaded but the obj id
            # has been set.  in the case of initial DAV PUT
            filename = extract_filename(file, default=self.filename or self.getId())
            if not filename:
                raise ValueError, 'Filename could not be found'
            filename = clean_filename(filename)
            # Update filename so that computing the new filename
            # uses the uploaded file's filename
            self._updateProperty('filename', filename)
            self.content_type = getMimeTypeFromFile(filename)
            fp = fspath(self)
            self._updateProperty('filepath', fp)
            # And then save the (possibly generated) filename
            self._updateProperty('filename', os.path.basename(fp))
            ftool.checkForReusedPath(fp)

            if not hasattr(file, 'read'):
                # Not a file-like thingie
                file = StringIO(file)
            copy_file(file, fp)

            # if moving from one repository to another, delete the
            # file in the old repository
            if (oldRepository and
                oldRepository != newRepository and
                oldFilepath != fp):
                ftool.cleanupFiles(oldFilepath)

        if delete_policy:
            self._updateProperty('deletePolicy', delete_policy)
        self.reindexObject()

InitializeClass(ManagedFileMixin)

class CMFManagedFile(ManagedFileMixin, PortalContent, DefaultDublinCoreImpl):
    """ A CMFManagedFile - stores documents on a file system rather than
    inside of the object database, provides policy-based management of
    the file system files.
    """

    meta_type = 'CMF ManagedFile'

    # Declarative security
    security = ClassSecurityInfo()
    security.declareObjectProtected(View)

    __implements__ = (ManagedFileMixin.__implements__,
                      PortalContent.__implements__,
                      DefaultDublinCoreImpl.__implements__)

    def __init__(self, id, title='', description='', fs_path=''):
        ManagedFileMixin.__init__(self, id, title, description, fs_path)
        DefaultDublinCoreImpl.__init__(self)
        self.setTitle(title)
        self.setDescription(description)

    def __call__(self, client=None, REQUEST=None, **kw):
        """ """
        return PortalContent.__call__(self)

    def manage_beforeDelete(self, item, container):
        ManagedFileMixin.manage_beforeDelete(self, item, container)
        PortalContent.manage_beforeDelete(self, item, container)

InitializeClass(CMFManagedFile)
