"""
ExternalFileBatch.py

A Factory for creating batches of ExternalFile instances in one go.
This class can be subclassed so that other kinds of similar objects
can be created in a batch.  cf. CVSFile.CVSFileBatch

Author: <a href="mailto:cstrong@arielpartners.com">Craeg Strong</a>
Release: 1.2
"""

# python builtins
import os.path, string, sys, traceback
from pprint import pprint
from cStringIO import StringIO

# Zope builtins
from Globals import MessageDialog
from OFS.ObjectManager import BadRequestException
from Acquisition import aq_parent, aq_base
from ZODB.POSException import ConflictError
from Products.ExternalFile.FileUtils import safe_path_join
from Products.CMFCore.utils import getToolByName

from content import addCMFManagedFile as addObject
from content import CMFManagedFile
from config import DELETE_MANUAL, TOOL_NAME

EXC_LIMIT = 4
def safe_call(func, messages, *args, **kw):
    limit = kw.get('limit', EXC_LIMIT)
    if kw.has_key('limit'):
        del kw['limit']
    try:
        return func(*args, **kw)
    except ConflictError:
        raise
    except KeyboardInterrupt:
        raise
    except:
        out = StringIO()
        pprint(func, out)
        pprint(args, out)
        pprint(kw, out)
        traceback.print_exc(limit=limit, file=out)
        messages.append(out.getvalue())
        del out
    return

################################################################
# Constructors
################################################################

# derived classes must create their own version of this function
def manage_addBatchViaGui(folder, target_dir = '', basedir='', recursive=None,
                          illegalChars = '', dirNames = '', dirPrefixes = '',
                          dirSuffixes = '', fileNames = '',
                          filePrefixes = '', fileSuffixes = '',
                          REQUEST=None):
    """
    Factory method to create a batch of instances of ExternalFile,
    called from GUI in the ZMI
    """
    results = '<h2>External File Batch Create Results:</h2>'

    try:
        batchAdd = folder.manage_addBatch
        results = results + batchAdd(target_dir, basedir, recursive,
                                     illegalChars, dirNames, dirPrefixes,
                                     dirSuffixes, fileNames, filePrefixes,
                                     fileSuffixes, 'default', DELETE_MANUAL)
    except Exception, e:
        results = results + str(e)

    return MessageDialog(title   = 'results',
                         message = results.replace('\n','<br/>'),
                         action  = 'manage_main')

def manage_addBatch(folder, target_dir='', recursive=None,
                    illegalChars = '', dirNames = '', dirPrefixes = '',
                    dirSuffixes = '', fileNames = '',
                    filePrefixes = '', fileSuffixes = '',
                    target_repository = 'default',
                    delete_policy = DELETE_MANUAL,
                    folder_type='Folder',
                    file_type='ManagedFile'):
    """ Factory method to actually create a batch of instances of
    ExternalFile.  This is designed to be called from PythonScripts.
    """
    importer = BatchImporter(folder=folder,
                             target_dir=target_dir,
                             recursive=recursive,
                             illegalChars=illegalChars,
                             dirNames=dirNames,
                             dirPrefixes=dirPrefixes,
                             dirSuffixes=dirSuffixes,
                             fileNames=fileNames,
                             filePrefixes=filePrefixes,
                             fileSuffixes=fileSuffixes,
                             target_repository=target_repository,
                             delete_policy=delete_policy,
                             folder_type=folder_type,
                             file_type=file_type)
    return importer()

trans = string.maketrans('', '')
trans = string.maketrans(trans[128:], '_' * len(trans[128:]))
def clean_name(str):
    return string.translate(str, trans)

class BatchImporter:

    # omit version control files
    omittedDirNames = ('CVS', '.svn')

    # omit directories like .ssh, .mozilla
    omittedDirPrefixes = ('.', '_svn')

    omittedDirSuffixes = ()

    omittedFileNames = ()

    # omit UNIX dot files or UNIX pipes
    # and Zope doesn't like things that begin with
    # underscores (underscores are ok, just not as first
    # character)
    omittedFilePrefixes = ('.', '@', '_', '#')

    # omit backup files
    omittedFileSuffixes = ('~', '.swp')

    # Chars in the filename that are illegal for Zope URLs
    # these characters are automatically elided
    illegalChars = ('@','%',':','!','`','^','*','=','+','|',
                    '\\','/','<','>','?','[',']','{','}','&')

    def __init__(self, folder, target_dir='', recursive=None,
                 illegalChars='', dirNames='', dirPrefixes='',
                 dirSuffixes='', fileNames='', filePrefixes='',
                 fileSuffixes = '', target_repository='default',
                 delete_policy=DELETE_MANUAL,
                 folder_type='Folder',
                 file_type='ManagedFile'):

        tool = getToolByName(folder, TOOL_NAME)
        repo = tool.getRepositoryInfo(target_repository, default=None)
        if repo is None:
            raise ValueError, 'Invalid Repository: %s' % target_repository
        base_dir, fs_policy, fn_policy = repo

        dirname = safe_path_join(base_dir, target_dir)
        if (not dirname or not os.path.exists(dirname) or
            not os.path.isdir(dirname)):
            message = ('ERROR: directory %s does not exist or is '
                       'not readable. No action taken.') % `dirname`
            raise Exception(message)
        self.committed = 0        # number of stub objects registered
        self.trnxthreshold = 1000 # number of commits till subtrnx commit
        self.messages = []
        self.omittedDirNames     = tuple(dirNames.split())
        self.omittedDirPrefixes  = tuple(dirPrefixes.split())
        self.omittedDirSuffixes  = tuple(dirSuffixes.split())
        self.omittedFileNames    = tuple(fileNames.split())
        self.omittedFilePrefixes = tuple(filePrefixes.split())
        self.omittedFileSuffixes = tuple(fileSuffixes.split())
        self.dirname = dirname
        self.base_dir = base_dir
        self.target_repository = target_repository
        self.delete_policy = delete_policy
        self.folder = folder
        self.recursive = recursive
        self.folder_type = folder_type
        self.file_type = file_type


    def __call__(self):
        path = self.dirname
        dirname = os.path.basename(path)
        
        folder = self.folder
        base_dir = self.base_dir
        args = (folder, base_dir, self.target_repository, self.delete_policy)
        safe_call(self.nodeaction, self.messages, args, path,
                os.listdir(path))

        return '\n'.join(self.messages)

    def nodeaction(self, options, dirname, filelist):
        """
        This method is called at every node of the file tree to create
        objects and folders.
        """
        messages = self.messages
        omittedDirNames = self.omittedDirNames
        illegalChars = self.illegalChars

        folder, base_dir, target_repository, delete_policy = options

        if os.path.basename(dirname) in omittedDirNames:
            messages.append('Skipping directory ' + dirname +
                            ' ...matches omitted name')
            return

        (files, dirs) = ([],[])
        for f in filelist:
            ff = safe_path_join(dirname, f)
            e =  filter(lambda x, baddies=illegalChars: x not in baddies, f)
            e = clean_name(e)
            if os.path.isfile(ff):
                files.append((ff, e))
            if os.path.isdir(ff):
                dirs.append((ff, e))

        messages.append('Processing %s files %s dirs %s' %
                        (`dirname`, len(files), len(dirs)))
        filelist[:] = []

        for filepath, name in files:
            prefix = os.path.commonprefix((filepath, base_dir))
            target_filepath = filepath[len(prefix)+len(os.path.sep):]
            safe_call(self.createObjectFromFile, messages,
                      folder=folder, target_filepath=target_filepath,
                      object_id=name, base_dir=base_dir,
                      target_repository=target_repository,
                      delete_policy=delete_policy)
            if self.committed % self.trnxthreshold:
                get_transaction().commit(1)

        for directory, name in dirs:
            safe_call(self.createFolderFromDirectory,
                      self.messages,
                      folder=folder, directory=directory,
                      object_id=name, base_dir=base_dir,
                      target_repository=target_repository,
                      delete_policy=delete_policy)
            if self.committed % self.trnxthreshold:
                get_transaction().commit(1)

    def createObjectFromFile(self, folder, target_filepath, object_id,
                             base_dir, target_repository,
                             delete_policy):
        """Factory to create objects from files in the file system.
        """


        filename = os.path.basename(target_filepath)
        fullpath = safe_path_join(base_dir, target_filepath)

        messages = self.messages
        omittedFileNames = self.omittedFileNames
        omittedFilePrefixes = self.omittedFilePrefixes
        omittedFileSuffixes = self.omittedFileSuffixes

        for name in omittedFileNames:
            if filename == name:
                messages.append('Skipping file: '+ fullpath +
                                 ' ...matches omitted name ')
                return

        for prefix in omittedFilePrefixes:
            if filename[:len(prefix)] == prefix:
                messages.append('Skipping file: '+ fullpath +
                                 ' ...matches omitted prefix')
                return

        for suffix in omittedFileSuffixes:
            if filename[-1 * len(suffix):] == suffix:
                messages.append('Skipping file: ' + fullpath +
                                 ' ...matches omitted suffix')
                return

        if os.path.isdir(fullpath):
            messages.append('Skipping: ' + fullpath +
                             ' ...because it is a directory, not a file')
            return

        if not os.path.isfile(fullpath):
            messages.append('Skipping file: ' + fullpath +
                             ' ...not regular file')
            return

        if not os.access(fullpath, os.R_OK):
            messages.append('Skipping file: ' + fullpath +
                            ' ...not a readable file')
            return

        try:
            kw = dict(target_filepath=target_filepath,
                      repository=target_repository,
                      delete_policy=delete_policy)
            folder.invokeFactory(self.file_type, object_id, **kw)
            ob = folder._getOb(object_id)
            messages.append('Created File: ' + object_id +
                            ' pointing to file: ' + fullpath)
            self.committed += 1

        except BadRequestException, msg:
            if not str(msg).find('already in use') == -1:
                messages.append('Skipping file: ' + fullpath +
                                ' ...duplicate Zope ID ' + object_id)
            else:
                raise

    def createFolderFromDirectory(self, folder, directory, object_id, base_dir,
                                  target_repository, delete_policy):
        """
        Factory for creating Zope folders from directories in the file
        system
        """

        messages = self.messages
        omittedDirNames = self.omittedDirNames
        omittedDirPrefixes = self.omittedDirPrefixes
        omittedDirSuffixes = self.omittedDirSuffixes

        dirname  = str(os.path.basename(directory))
        fullpath = directory

        if dirname in omittedDirNames:
            messages.append('Skipping directory ' + fullpath +
                            ' ...matches omitted name')
            return

        for prefix in omittedDirPrefixes:
            if dirname[:len(prefix)] == prefix:
                messages.append('Skipping directory: '+ fullpath +
                                 ' ...matches omitted prefix')
                return

        for suffix in omittedDirSuffixes:
            if dirname[ -1 * len(suffix):] == suffix:
                messages.append('Skipping directory: '+ fullpath +
                                 ' ...matches omitted suffix')
                return

        try:
            folder.invokeFactory(self.folder_type, object_id)
            messages.append('Created Folder: ' + object_id +
                             ' from directory ' + fullpath)
            self.committed += 1
            sub = folder._getOb(object_id)
            args = (sub, base_dir, target_repository, delete_policy)
            if self.recursive:
                safe_call(self.nodeaction, self.messages, args, directory,
                          os.listdir(fullpath))
            #os.path.walk(fullpath, self.nodeaction, (sub, base_dir,
            #                                    target_repository,
            #                                    delete_policy))
        except BadRequestException, msg:
            if not str(msg).find('already in use') == -1:
                messages.append('Skipping directory: ' + fullpath +
                                ' ...duplicate Zope ID ' + dirname)
            else:
                raise

# EOF ExternalFileBatch.py
